  /**********************************************************************
  *                            LockFileReader                            *
  ************************************************************************
  * Copyright (C) by Michael Loesler, http://derletztekick.com           *
  *                                                                      *
  * This program is free software; you can redistribute it and/or modify *
  * it under the terms of the GNU General Public License as published by *
  * the Free Software Foundation; either version 3 of the License, or    *
  * (at your option) any later version.                                  *
  *                                                                      *
  * This program is distributed in the hope that it will be useful,      *
  * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
  * GNU General Public License for more details.                         *
  *                                                                      *
  * You should have received a copy of the GNU General Public License    *
  * along with this program; if not, see <http://www.gnu.org/licenses/>  *
  * or write to the                                                      *
  * Free Software Foundation, Inc.,                                      *
  * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
  *                                                                      *
   **********************************************************************/

package com.derletztekick.tools.io;
import java.io.File;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

/**
 * Klasse zum Einlesen von ASCII-Dateien. Die Dateien werden Zeilenweise in
 * eine Liste geschrieben. Beim Einlesen der Datei kann diese nicht von einer
 * anderen Anwendung veraendert werden, sie wird gesperrt!
 * 
 * @author Michael Loesler <derletztekick.com>
 * @see <http://aktuell.de.selfhtml.org/artikel/programmiertechnik/dateisperren/java.htm>
 *
 */

public abstract class LockFileReader {
	private boolean isComplete = false, isInterrupt = false;
	private File sourceFile = null;
	private String ignoreStartString = new String();
	public static final String UTF8_BOM = "\uFEFF";
	/**
	 * Konstruktor, der den Namen der Datei erwartet (wird weitergegeben)
	 * @param fileName Dateiname (mit Pfad)
	 */
	public LockFileReader(String fileName) {
		this(new File(fileName));
	}

	/**
	 * Konstruktor, der einer File-Objekt erwartet
	 * @param sf Quelldatei
	 */
	public LockFileReader(File sf) {
		this.sourceFile = sf;
	}
  
	/**
	 * Diese Funktion wird beim zweilenweisen Einlesen
	 * der Datei pro Zeile aufgerufen. Erbene Funktionen
	 * muessen diese Methode zum Zerlegen der Zeichenkette
	 * sinnvoll ueberschreiben
	 * 
	 * @param line
	 */
	public abstract void parse(String line);  
	
	/**
	 * Gibt an, ob die Datei vollstaendig gelesen wurde
	 * @return isComplete
	 */
	public boolean isComplete(){
		return this.isComplete;
	}
	
	/**
	 * Legt die Zeichenkette fest, die beim Einlesen eine Zeile in der Datei markiert, 
	 * die nicht gelsen werden soll zB Kommentare oder aehnliches.
	 * 
	 * @param str Zeichenkette, die ueberlesen werden soll
	 */
	public void ignoreLinesWhichStartWith(String str) {
		this.ignoreStartString = str;
	}
  
	/**
	 * liest die Quelldatei sf zeilenweise in eine Liste ein und gibt im Erfolgsfall <code>true</code> zurueck
	 * @return isComplete
	 */
	public boolean readSourceFile(){
		if (this.sourceFile == null || !this.sourceFile.exists() || 
    		!this.sourceFile.isFile() || !this.sourceFile.canRead())
			return this.isComplete = false;
		BufferedReader reader = null;
		try {
			FileInputStream is = new FileInputStream( this.sourceFile );
			is.getChannel().lock(0, Long.MAX_VALUE, true);
			reader = new BufferedReader(new InputStreamReader(is, "UTF-8"));
			String line = null;
			boolean isFirstLine = true;
			while (( line = reader.readLine()) != null && !this.isInterrupt) {
				if (isFirstLine && line.startsWith(UTF8_BOM)) {
					line = line.substring(1); 
					isFirstLine = false;
				}
				if (!line.trim().isEmpty() && (this.ignoreStartString.length() == 0
                        || !line.startsWith( this.ignoreStartString )))
                    this.parse(line);
			}
			this.isInterrupt = false;
		}
		catch (IOException e) {
			System.err.println("Fehler, Datei "+this.sourceFile.getName()+" konnte nicht gelesen werden!\n");
			e.printStackTrace();
			return this.isComplete = false;
		}
		finally {
			if (reader!=null)
				try {
					reader.close();
				}
				catch(IOException e) {
					System.err.println("Fehler beim Schliessen des Lesestroms!\n");
					e.printStackTrace();
				}
		}
		return this.isComplete = true;
	}
	
	/**
	 * Bricht den Lesevorgang ab.
	 */
	public void interrupt() {
		this.isInterrupt = true;
	}
	
	/**
	 * Liefert die einzulesende Datei zurueck
	 * @return file
	 */
	public File getFile() {
		return this.sourceFile;
	}
	
	/**
	 * Setzt eine (neue) Datei, die einzulesen ist.
	 * @param file
	 */
	public void setFile(File file) {
		this.sourceFile = file;
	}
  
}
