package com.derletztekick.tools.geodesy.distribution;

/**
 * 
 * @author Michael.Loesler
 * @see <http://www.freecx.co.uk/bcd32/BitStatistic/src/ENT/chisq.c>
 */
public class Normal {
	private final static double Z_MAX = 10.0;

	private Normal() {}
	
	public static double norminv(double p) {
		return norminv(p, 0.0, 1.0);
	}
	
	public static double norminv(double p, double mu, double sigma) {
		if (p <= 0)
			return Double.NEGATIVE_INFINITY;

		if (p >= 1)
			return Double.POSITIVE_INFINITY;

		if (sigma <= 0)
			return mu;


		double q, r, val;
		q = p - 0.5;

		if (Math.abs(q) <= .425) {/* 0.075 <= p <= 0.925 */
			r = .180625 - q * q;
			val =
					q * (((((((r * 2509.0809287301226727 +
							33430.575583588128105) * r + 67265.770927008700853) * r +
							45921.953931549871457) * r + 13731.693765509461125) * r +
							1971.5909503065514427) * r + 133.14166789178437745) * r +
							3.387132872796366608)
							/ (((((((r * 5226.495278852854561 +
									28729.085735721942674) * r + 39307.89580009271061) * r +
									21213.794301586595867) * r + 5394.1960214247511077) * r +
									687.1870074920579083) * r + 42.313330701600911252) * r + 1);
		}
		else { /* closer than 0.075 from {0,1} boundary */

			if (q > 0)
				r = 1 - p;
			else
				r = p;

			r = Math.sqrt(-Math.log(r));

			if (r <= 5) { /* <==> min(p,1-p) >= exp(-25) ~= 1.3888e-11 */
				r += -1.6;
				val = (((((((r * 7.7454501427834140764e-4 +
						.0227238449892691845833) * r + .24178072517745061177) *
						r + 1.27045825245236838258) * r +
						3.64784832476320460504) * r + 5.7694972214606914055) *
						r + 4.6303378461565452959) * r +
						1.42343711074968357734)
						/ (((((((r *
								1.05075007164441684324e-9 + 5.475938084995344946e-4) *
								r + .0151986665636164571966) * r +
								.14810397642748007459) * r + .68976733498510000455) *
								r + 1.6763848301838038494) * r +
								2.05319162663775882187) * r + 1);
			}
			else { /* very close to  0 or 1 */
				r += -5;
				val = (((((((r * 2.01033439929228813265e-7 +
						2.71155556874348757815e-5) * r +
						.0012426609473880784386) * r + .026532189526576123093) *
						r + .29656057182850489123) * r +
						1.7848265399172913358) * r + 5.4637849111641143699) *
						r + 6.6579046435011037772)
						/ (((((((r *
								2.04426310338993978564e-15 + 1.4215117583164458887e-7) *
								r + 1.8463183175100546818e-5) * r +
								7.868691311456132591e-4) * r + .0148753612908506148525)
								* r + .13692988092273580531) * r +
								.59983220655588793769) * r + 1);
			}

			if (q < 0.0)
			{
				val = -val;
			}
		}

		return mu + sigma * val;
	}

	public static double normcdf(double z, double mu, double std) {
		return normcdf((z-mu)/std);
	}

	public static double normcdf(double z) {
		double y, x, w;

		if (z == 0.0) {
			x = 0.0;
		} else {
			y = 0.5 * Math.abs(z);
			if (y >= (Z_MAX * 0.5)) {
				x = 1.0;
			} else if (y < 1.0) {
				w = y * y;
				x = ((((((((0.000124818987 * w
						-0.001075204047) * w +0.005198775019) * w
						-0.019198292004) * w +0.059054035642) * w
						-0.151968751364) * w +0.319152932694) * w
						-0.531923007300) * w +0.797884560593) * y * 2.0;
			} else {
				y -= 2.0;
				x = (((((((((((((-0.000045255659 * y
						+0.000152529290) * y -0.000019538132) * y
						-0.000676904986) * y +0.001390604284) * y
						-0.000794620820) * y -0.002034254874) * y
						+0.006549791214) * y -0.010557625006) * y
						+0.011630447319) * y -0.009279453341) * y
						+0.005353579108) * y -0.002141268741) * y
						+0.000535310849) * y +0.999936657524;
			}
		}
		return (z > 0.0 ? ((x + 1.0) * 0.5) : ((1.0 - x) * 0.5));
	}
	
	public static double cumulative_standard(double x) {
		return cumulative(x, 0.0, 1.0, true, false);
	}
	
	public static double cumulative(double x, double mu, double sigma) {
		return cumulative(x, mu, sigma, true, false);
	}
	
	public static double cumulative(double x, double mu, double sigma, boolean lower_tail, boolean log_p) {
		double p = Normal.normcdf(x, mu, sigma);
		if (!lower_tail)
			p = 1.0 - p;
		if (log_p)
			p = Math.log(p);
		return p;
	}
	
	public static double quantile(double p, double mu, double sigma, boolean lower_tail, boolean log_p) {
		if (log_p)
			p = Math.pow(Math.E, p);
		if (!lower_tail)
			p = 1.0 - p;
		return Normal.norminv(p, mu, sigma);
	}

	public static void main(String args[]) {
		double x = 15.5;
		double mu = 17.8;
		double sigma = 10;
		double p = -0.0010005049704290754;
//		double p = -6.90775064678418;

		boolean lower_tail = true;
		boolean log_p = true;
		
		System.out.println(jdistlib.Normal.cumulative(x, mu, sigma, lower_tail, log_p));
		System.out.println(Normal.cumulative(x, mu, sigma, lower_tail, log_p));
		
		System.out.println(jdistlib.Normal.cumulative(x, mu, sigma));
		System.out.println(Normal.cumulative(x, mu, sigma));
		
		System.out.println(jdistlib.Normal.cumulative_standard(x));
		System.out.println(Normal.cumulative_standard(x));
		
		System.out.println(jdistlib.Normal.quantile(p, mu, sigma, lower_tail, log_p));
		System.out.println(Normal.quantile(p, mu, sigma, lower_tail, log_p));

	}
}
