package com.derletztekick.tools.geodesy.distribution;

import com.derletztekick.tools.geodesy.Constant;

public class NonCentralChiSquare {
	private final static int ITERMAX = 500;
	private static final double EPSILON = Math.sqrt(Constant.EPS);
	
	private NonCentralChiSquare() {}
	
	public static double ncchi2cdf(double x, double df, double ncp) {
		double lower = 0.0;
		double upper = 1.0;
		int i = 0;
		double mean = 0.5*(upper + lower);
		double t = ncchi2inv(mean, df, ncp);
		while ((Math.abs(upper - lower) > EPSILON || Math.abs(t - x) > EPSILON) && i++ < ITERMAX) {
			t = ncchi2inv(mean, df, ncp);
			if (t < x)
				upper = mean;
			else
				lower = mean;
			mean = 0.5*(upper + lower);			
		}
		return mean;
	}
	
	public static double ncchi2inv(double alpha, double df, double ncp)  {
		if (alpha <= 0.0)
			return 0.0;
		else if (alpha >= 1.0)
			return Double.POSITIVE_INFINITY;
		
		double x2 = df + 2.0*ncp;
		double x3 = df + 3.0*ncp;
		double b = -ncp*ncp / x3;
		double c = x3 / x2;
		double f = x2*x2*x2 / x3/x3;
		
		return b + c * ChiSquare.chi2inv(alpha, f);
	}
	
	public static double cumulative(double x, double df, double ncp, boolean lower_tail, boolean log_p) {
		double p = NonCentralChiSquare.ncchi2cdf(x, df, ncp);
		if (lower_tail)
			p = 1.0 - p;
		if (log_p)
			p = Math.log(p);
		return p;
	}
	
	public static double quantile(double p, double df, double ncp, boolean lower_tail, boolean log_p) {
		if (log_p)
			p = Math.pow(Math.E, p);
		if (lower_tail)
			p = 1.0 - p;
		return NonCentralChiSquare.ncchi2inv(p, df, ncp);
	}
	
	
	public static void main(String args[]) {
//		double p = -0.0010005049704290754;
		double p = -6.90775064678418;
		double df = 250;
		double ncp = 25;
		boolean lower_tail = true;
		boolean log_p = true;
		System.out.println(jdistlib.NonCentralChiSquare.quantile(p, df, ncp, lower_tail, log_p));
		System.out.println(NonCentralChiSquare.quantile(p, df, ncp, lower_tail, log_p));
		
		System.out.println(jdistlib.NonCentralChiSquare.cumulative(205.43654037713424, df, ncp, lower_tail, log_p));
		System.out.println(NonCentralChiSquare.cumulative(205.43654037713424, df, ncp, lower_tail, log_p));
		
	}
}
