 /**********************************************************************
 * Copyright (C) by Michael Loesler, http://derletztekick.com           *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, see <http://www.gnu.org/licenses/>  *
 * or write to the                                                      *
 * Free Software Foundation, Inc.,                                      *
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
 *                                                                      *
  **********************************************************************/

package com.derletztekick.tools.geodesy.distribution;

import java.util.Random;

public class Distribution {
	private final static Random RANDOM_GENERATOR = new Random();
	
	/**
	 * Rechteckverteilung fuer einen gegebenen Erwartungswert mean und einen Abstand [-a a]
	 * @param mean
	 * @param a
	 * @return rand
	 */
	public static double randUniformBoundary(double mean, double a) {
		double r = RANDOM_GENERATOR.nextDouble();
		return (r * 2.0 * a) + mean - a;
	}
	
	/**
	 * Rechteckverteilung fuer einen gegebenen Erwartungswert mean und eine Unter- bzw. Obergrenze [a b] (nicht symmetrisch)
	 * @param mean
	 * @param a
	 * @param b
	 * @return rand
	 */
	public static double randUniformBoundary(double mean, double a, double b) {
		double r = RANDOM_GENERATOR.nextDouble();
		double bounds = b - a;
		return (r * bounds) + mean - a;
	}
		
	/**
	 * Dreiecksverteilung fuer einen gegebenen Erwartungswert mean und einen Abstand [-a a]
	 * @param mean
	 * @param a
	 * @return rand
	 */
	public static double randTriangularBoundary(double mean, double a) {
		double r1 = RANDOM_GENERATOR.nextDouble();
		double r2 = RANDOM_GENERATOR.nextDouble();
		return (r1 + r2) * a + mean - a;
	}
	
	/**
	 * Dreiecksverteilung fuer einen gegebenen Erwartungswert mean und eine Unter- bzw. Obergrenze [a b] (nicht symmetrisch)
	 * @param mean
	 * @param a
	 * @param b
	 * @return rand
	 */
	public static double randTriangularBoundary(double mean, double a, double b) {
	    double whole = b - a;
	    double r = RANDOM_GENERATOR.nextDouble();
	    if (r <= (a / whole))
	        return a + Math.sqrt(r * whole * a);
	    else
	        return b - Math.sqrt((1.0 - r) * whole * b);
	}
	
	/**
	 * Normalverteilung fuer einen gegebenen Erwartungswert mean und eine Standardabweichung sigma
	 * @param mean
	 * @param sigma
	 * @return rand
	 */
	public static double randNormal(double mean, double sigma) {
		double r = RANDOM_GENERATOR.nextGaussian();
		return sigma * r + mean;
	}
	
	/**
	 * Dreiecksverteilung fuer einen gegebenen Erwartungswert mean und eine Standardabweichung sigma
	 * @param mean
	 * @param sigma
	 * @return rand
	 */
	public static double randTriangular(double mean, double sigma) {
		double a = Math.sqrt(6.0 * sigma * sigma);
		return randTriangularBoundary(mean, a);
	}
	
	/**
	 * Dreiecksverteilung fuer einen gegebenen Erwartungswert mean und eine Standardabweichung sigma
	 * @param mean
	 * @param sigma
	 * @return rand
	 */
	public static double randUniform(double mean, double sigma) {
		double a = Math.sqrt(3.0 * sigma * sigma);
		return randUniformBoundary(mean, a);
	}
	
}
