package com.derletztekick.tools.geodesy.distribution;

import com.derletztekick.tools.geodesy.Constant;

public class ChiSquare {
	private final static int ITERMAX = 500;
	private static final double EPSILON = Math.sqrt(Constant.EPS);
	
	private ChiSquare() {}
	
	private static double gammln(double xx){
	    // Returns the value ln[GAMMA(xx)] for xx > 0.
		double x = xx,
	           y = xx,
	           tmp = x + 5.5,
	           ser=1.000000000190015;

		double cof[] = {
	                      76.18009172947146,
	                     -86.50532032941677,
	                      24.01409824083091,
	                     -1.231739572450155,
	                      0.1208650973866179E-2,
	                     -0.5395239384953e-5
	                   };

		tmp -= (x+0.5)*Math.log(tmp);

		for (int j=0; j<6; j++)
			ser += cof[j]/(double)++y;
		return -tmp+Math.log(2.5066282746310005*ser/x);
	}
	
	private static double Gser(double X, double A) {
		double T9=1.0/A;
		double G=T9;
		int i=1;
		while (T9 > G * EPSILON && i < ITERMAX) {
			T9 *= X/(A+i);
			G=G+T9;
			i++;
		}
		G *= Math.exp(A*Math.log(X)-X-gammln(A));

		return G;
	}
	
	private static double Gcf(double X, double A) {
		double A0=0.0;
		double B0=1.0;
		double A1=1.0;
		double B1=X;
		double AOLD=0.0;
		int i=0;
		while (Math.abs((A1-AOLD)/A1) > EPSILON && i < ITERMAX) {
			AOLD=A1;
			i++;
			A0 = A1+(i-A)*A0;
			B0 = B1+(i-A)*B0;
			A1 = X*A0+i*A1;
			B1 = X*B0+i*B1;
			A0 = A0/B1;
			B0 = B0/B1;
			A1 = A1/B1;
			B1 = 1.0;
		}
		return 1.0 - Math.exp(A*Math.log(X)-X-gammln(A))*A1;
	}
	
	private static double Gammacdf(double x, double a) {
		double GI = 0;
		if (x<=0)
			GI=0;
		else if (x < a+1) 
			GI = Gser(x,a);
		else
			GI = Gcf(x,a);

		return GI;
	}
	
	public static double chi2cdf(double x, double df) {
		return Gammacdf(x/2.0,df/2.0);
	}
		
	public static double chi2inv(double p, double df) {
		double lower = 0.0;
		double upper = Double.POSITIVE_INFINITY;
		double mean;

		if (p <= 0.0)
			return 0.0;
		else if (p >= 1.0)
			return upper;

		int i=0;
		mean = df / Math.sqrt(p);
		double t = 1.0 - chi2cdf(mean, df);
		while ((Math.abs(upper - lower) > EPSILON || Math.abs(t - p) > EPSILON) && i++ < ITERMAX) {
			t = 1.0 - chi2cdf(mean, df);
			if (t < p) {
				upper = mean;
			} else {
				lower = mean;
			}
			mean = 0.5 * (upper + lower);
		}
		return mean;
	}
	
	public static double cumulative(double x, double df, boolean lower_tail, boolean log_p) {
		double p = ChiSquare.chi2cdf(x, df);
		if (!lower_tail)
			p = 1.0 - p;
		if (log_p)
			p = Math.log(p);
		return p;
	}
	
	public static double quantile(double p, double df, boolean lower_tail, boolean log_p) {
		if (log_p)
			p = Math.pow(Math.E, p);
		if (lower_tail)
			p = 1.0 - p;
		return ChiSquare.chi2inv(p, df);
	}
	
	public static void main(String args[]) {
		double p = 0.0010005049704290754;
//		double p = -6.90775064678418;
		double df = 250;
		boolean lower_tail = true;
		boolean log_p = false;
		System.out.println(jdistlib.ChiSquare.quantile(p, df, lower_tail, log_p));
		System.out.println(ChiSquare.quantile(p, df, lower_tail, log_p));

		System.out.println(jdistlib.ChiSquare.cumulative(324.8324, df, lower_tail, log_p));
		System.out.println(ChiSquare.cumulative(324.8324, df, lower_tail, log_p));
	}
}
