 /**********************************************************************
 * Copyright (C) by Michael Loesler, http://derletztekick.com           *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, see <http://www.gnu.org/licenses/>  *
 * or write to the                                                      *
 * Free Software Foundation, Inc.,                                      *
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
 *                                                                      *
  **********************************************************************/

package com.derletztekick.tools.geodesy.delaunay;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

public class Triangle2D {
	private final Circle2D circle;

	private Set<Triangle2D> triangles = new LinkedHashSet<Triangle2D>(3);
	private Map<String, Point2D> trianglePoints = new LinkedHashMap<String, Point2D>(3);
	
	public Triangle2D(Point2D pi, Point2D pj, Point2D pk) throws ArithmeticException {
		this.circle = new Circle2D(pi, pj, pk);
		if (!this.circle.isCircle())
			throw new ArithmeticException(this.getClass().getSimpleName() + " Fehler, Punkte liegen nicht " +
					"auf einem Kreis -> kein regulaeres Dreieck!\n" + pi + "\n" + pj + "\n" + pk);
		this.trianglePoints.put(pi.getId(), pi);
		this.trianglePoints.put(pj.getId(), pj);
		this.trianglePoints.put(pk.getId(), pk);
	}
	
	public Set<Triangle2D> getNeighbours() {
		return this.triangles;
	}
	
	/**
	 * Liefert true, wenn es ein regulaeres Dreick ist (nicht alle Punkte auf einer Geraden)
	 * @return isTriangle
	 */
	public boolean isRegularTriangle() {
		return this.circle.isCircle();
	}
	
	/**
	 * Gibt den Umkreis zurueck.
	 * @return circumCircle
	 */
	public Circle2D getCircumCircle() {
		return this.circle;
	}
	
	/**
	 * Liefert die drei Eckpunkte des Dreiecks.
	 * @return points
	 */
	public Point2D[] getTrianglePoints() {
		return this.circle.getCirclePoints();
	}
	
	/**
	 * Fuegt ein (angrenzendes) Nachbardreieck hinzu
	 * @param triangle
	 */
	public void addNeighbour(Triangle2D triangle) {
		if (triangle != null && !this.equals(triangle))
			this.triangles.add(triangle);
	}
	
	/**
	 * Entfernt ein Nachbardreieck.
	 * @param triangle
	 */
	public void removeNeighbour(Triangle2D triangle) {
		this.triangles.remove(triangle);
	}
	
	/**
	 * Liefert <code>true</code> wenn es einer der drei Punkt des Dreiecks ist.
	 * @param p
	 * @return contains
	 */
	public boolean contains(Point2D p) {
		return this.trianglePoints.containsKey(p.getId());
	}
	
	/**
	 * Bestimmt, ob ein Punkt im Dreieck liegt == -1
	 * auf einer Seite des Dreiecks liegt == 0
	 * oder auerhalb ist == 1
	 */
	public int containsPoint(Point2D p) {
		if (this.trianglePoints.containsKey(p.getId()))
			return 0;
				
		Point2D[] points = this.circle.getCirclePoints();
		double p1x = points[0].getX();
		double p1y = points[0].getY();
		
		double p2x = points[1].getX();
		double p2y = points[1].getY();
		
		double p3x = points[2].getX();
		double p3y = points[2].getY();
		
		double px = p.getX();
		double py = p.getY();

        double det1=((p2x-p1x)*(py-p1y)-(p2y-p1y)*(px-p1x));
        double det2=((p3x-p2x)*(py-p2y)-(p3y-p2y)*(px-p2x)); 
        double det3=((p1x-p3x)*(py-p3y)-(p1y-p3y)*(px-p3x));
        
        // Punkt liegt auf Seite P1-P2 des Dreiecks
        if (det1 == 0 && (((p1x <= px && px <= p2x) || (p2x <= px && px <= p1x)) && ((p1y <= py && py <= p2y) || (p2y <= py && py <= p1y)))) {
        	return 0;
        }
        // Punkt liegt auf Seite P2-P3 des Dreiecks
        else if (det2 == 0 && (((p3x <= px && px <= p2x) || (p2x <= px && px <= p3x)) && ((p3y <= py && py <= p2y) || (p2y <= py && py <= p3y)))) {
        	return 0;
        }
        // Punkt liegt auf Seite P1-P3 des Dreiecks
        else if (det3 == 0 && (((p3x <= px && px <= p1x) || (p1x <= px && px <= p3x)) && ((p3y <= py && py <= p1y) || (p1y <= py && py <= p3y)))) {
        	return 0;
        }

        int sum = (int)Math.signum(det1) + (int)Math.signum(det2) + (int)Math.signum(det3); 
        
        return Math.abs(sum) == 3?-1:1;
	}
	
	/**
	 * Liefert das Nachbardreieck anhand zweier Punkte (einer Dreiecksseite).
	 * Sollte es kein Dreieck geben, liefert die Methode NULL zurueck.
	 * 
	 * @param pi
	 * @param pk
	 * @return triangle
	 */
	public Triangle2D getNeighbour(Point2D pi, Point2D pk) {
		for (Triangle2D triangle : this.triangles) {
			if (triangle.containsPoint(pi) < 1 && triangle.containsPoint(pk) < 1)
				return triangle;
		}
		return null;
	}
	
	/**
	 * Liefert das Nachbardreieck anhand eines Punktes, der sich auf der Kante befindet.
	 * Sollte es kein Dreieck geben, liefert die Methode NULL zurueck.
	 * 
	 * @param pi
	 * @return triangle
	 */
	public Triangle2D getNeighbour(Point2D p) {
		Point2D[] points = this.getEdge(p);
		if (points == null)
			return null;
		return this.getNeighbour(points[0], points[1]);
	}
	
	/**
	 * Liefert die Dreiecksseite anhand eines Punktes, der sich auf der Kante befindet.
	 * Liegt der Punkt auf keiner Seite, liefert die Methode NULL zurueck.
	 * 
	 * @param pi
	 * @return triangle
	 */
	public Point2D[] getEdge(Point2D p) {
		Point2D[] points = this.circle.getCirclePoints();
		double p1x = points[0].getX();
		double p1y = points[0].getY();
		
		double p2x = points[1].getX();
		double p2y = points[1].getY();
		
		double p3x = points[2].getX();
		double p3y = points[2].getY();
		
		double px = p.getX();
		double py = p.getY();
		
		double det1=((p2x-p1x)*(py-p1y)-(p2y-p1y)*(px-p1x));
        double det2=((p3x-p2x)*(py-p2y)-(p3y-p2y)*(px-p2x)); 
        double det3=((p1x-p3x)*(py-p3y)-(p1y-p3y)*(px-p3x));
        
        // Punkt liegt auf Seite P1-P2 des Dreiecks
        if (det1 == 0 && (((p1x <= px && px <= p2x) || (p2x <= px && px <= p1x)) && ((p1y <= py && py <= p2y) || (p2y <= py && py <= p1y)))) {
        	return new Point2D[] {points[0], points[1]};
        }
        // Punkt liegt auf Seite P2-P3 des Dreiecks
        else if (det2 == 0 && (((p3x <= px && px <= p2x) || (p2x <= px && px <= p3x)) && ((p3y <= py && py <= p2y) || (p2y <= py && py <= p3y)))) {
        	return new Point2D[] {points[1], points[2]};
        }
        // Punkt liegt auf Seite P1-P3 des Dreiecks
        else if (det3 == 0 && (((p3x <= px && px <= p1x) || (p1x <= px && px <= p3x)) && ((p3y <= py && py <= p1y) || (p1y <= py && py <= p3y)))) {
        	return new Point2D[] {points[0], points[2]};
        }
		return null;
	}
	
	@Override
	public String toString() {
		return this.getClass().getSimpleName() + " [" + Arrays.toString(this.getTrianglePoints()) + "]";
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Triangle2D other = (Triangle2D) obj;
		Point2D[] otherPoints = other.getCircumCircle().getCirclePoints();
		
		for (int i=0; i<otherPoints.length; i++) 
			if (!this.trianglePoints.containsKey(otherPoints[i].getId()))
				return false;
		
		return true;
	}
}
