 /**********************************************************************
 * Copyright (C) by Michael Loesler, http://derletztekick.com           *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, see <http://www.gnu.org/licenses/>  *
 * or write to the                                                      *
 * Free Software Foundation, Inc.,                                      *
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
 *                                                                      *
  **********************************************************************/

package com.derletztekick.tools.geodesy.delaunay;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class Tetrahedron {
	private final Sphere sphere;

	private Set<Tetrahedron> tetrahedrons = new LinkedHashSet<Tetrahedron>(4);
	private Map<String, Point3D> tetrahedronPoints = new LinkedHashMap<String, Point3D>(4);
	public Tetrahedron(Point3D pi, Point3D pj, Point3D pk, Point3D pl) throws ArithmeticException{
		this.sphere = new Sphere(pi, pj, pk, pl);
		this.tetrahedronPoints.put(pi.getId(), pi);
		this.tetrahedronPoints.put(pj.getId(), pj);
		this.tetrahedronPoints.put(pk.getId(), pk);
		this.tetrahedronPoints.put(pl.getId(), pl);
	}
	
	public Sphere getCircumSphere() {
		return this.sphere;
	}
	
	public Point3D[] getTetrahedronPoints() {
		return this.sphere.getSpherePoints();
	}
	
	public void addNeighbour(Tetrahedron tetrahedron) {
		if (tetrahedron != null)
			this.tetrahedrons.add(tetrahedron);
	}
	
	public void removeNeighbour(Tetrahedron tetrahedron) {
		this.tetrahedrons.remove(tetrahedron);
	}
	
//	public boolean containsPoint(Point2D p) {
//		return this.trianglePoints.containsKey(p.getId());
//	}
	
	/**
	 * Bestimmt, ob ein Punkt im Tetraeder liegt == -1
	 * auf einer Seite des Tetraeders liegt == 0
	 * oder auerhalb ist == 1
	 */
	// http://steve.hollasch.net/cgindex/geometry/ptintet.html
	public int containsPoint(Point3D p) {
		if (this.tetrahedronPoints.containsKey(p.getId()))
			return 0;
				
		Point3D[] points = this.sphere.getSpherePoints();
		int sum = 0;
		for (int j=0; j<5; j++) {
			double a[][] = new double[4][4];
			for (int i=0; i<4; i++) {
		    	double x = j==i?p.getX():points[i].getX();
		    	double y = j==i?p.getY():points[i].getY();
		    	double z = j==i?p.getZ():points[i].getZ();
		    	
		        a[i][0] = x;
		        a[i][1] = y;
		        a[i][2] = z;
		        a[i][3] = 1.0;
		    }
			sum += (int)Math.signum(this.determinant(a));
		}

		if (Math.abs(sum) == 5)
			return -2; // drin
		else if (Math.abs(sum) == 4)
			return -1; // auf Kante
		else if (Math.abs(sum) == 3)
			return 0; // auf Flaeche
		return 1;
	}
	
	/**
	 * Liefert die Nachbartetraeder anhand zweier Punkte (einer Kante).
	 * Sollte es kein Tetraeder geben, liefert die Methode NULL zurueck.
	 * 
	 * @param pi
	 * @param pk
	 * @return tetrahedrons
	 */
	public List<Tetrahedron> getNeighbour(Point3D pi, Point3D pk) {
		List<Tetrahedron> neighbours = new ArrayList<Tetrahedron>();
		for (Tetrahedron tetrahedron : this.tetrahedrons) {
			if (tetrahedron.containsPoint(pi) < 1 && tetrahedron.containsPoint(pk) < 1)
				neighbours.add(tetrahedron);
		}
		return neighbours.size() == 0?null:neighbours;
	}
	
	/**
	 * Liefert das Nachbartetraeder anhand dreier Punkte (einer Flche).
	 * Sollte es kein Tetraeder geben, liefert die Methode NULL zurueck.
	 * 
	 * @param pi
	 * @param pj
	 * @param pk
	 * @return tetrahedron
	 */
	public Tetrahedron getNeighbour(Point3D pi, Point3D pj, Point3D pk) {
		for (Tetrahedron tetrahedron : this.tetrahedrons) {
			if (tetrahedron.containsPoint(pi) < 1 && tetrahedron.containsPoint(pj) < 1  && tetrahedron.containsPoint(pk) < 1)
				return tetrahedron;
		}
		return null;
	}
	
	/**
	 * Liefert die Nachbartetraeder anhand eines Punktes, der sich auf der Kante befindet.
	 * Sollte es kein Tetraeder geben, liefert die Methode NULL zurueck.
	 * 
	 * @param pi
	 * @return tetrahedron
	 */
	public List<Tetrahedron> getNeighbour(Point3D p) {
		Point3D[] points = this.getEdge(p);
		if (points == null)
			return null;
		return this.getNeighbour(points[0], points[1]);
	}
	
	/**
	 * Liefert die Kante anhand eines Punktes, der sich auf der Kante befindet.
	 * Sollte es keine Kante geben, liefert die Methode NULL zurueck.
	 * 
	 * @param pi
	 * @return triangle
	 */
	public Point3D[] getEdge(Point3D p) {
		Point3D[] points = this.sphere.getSpherePoints();
		int j1 = -1;
		int j2 = -1;
		for (int j=0; j<4; j++) {
			double a[][] = new double[4][4];
			for (int i=0; i<4; i++) {
		    	double x = j==i?p.getX():points[i].getX();
		    	double y = j==i?p.getY():points[i].getY();
		    	double z = j==i?p.getZ():points[i].getZ();
		    	
		        a[i][0] = x;
		        a[i][1] = y;
		        a[i][2] = z;
		        a[i][3] = 1.0;
		    }
			// Bestimme die beiden Flaechen, in denen p liegt
			if (this.determinant(a) == 0) {
				if (j1<0)
					j1 = j;
				else if (j2<0) {
					j2 = j;
					break;
				}
			}
		}
		
		// Nichts gefunden
		if (j1 < 0 || j2 < 0)
			return null;
		
		Point3D[] p3ds = new Point3D[2];
		for (int i=0, j=0; i<4; i++) {
			if (i != j1 && i != j2)
				p3ds[j++] = points[i];
		}
		
		if ((p3ds[0].getX() <= p.getX() && p.getX() <= p3ds[1].getX() || p3ds[1].getX() <= p.getX() && p.getX() <= p3ds[0].getX()) &&
				(p3ds[0].getY() <= p.getY() && p.getY() <= p3ds[1].getY() || p3ds[1].getY() <= p.getY() && p.getY() <= p3ds[0].getY()) &&
				(p3ds[0].getZ() <= p.getZ() && p.getZ() <= p3ds[1].getZ() || p3ds[1].getZ() <= p.getZ() && p.getZ() <= p3ds[0].getZ())) {
			return p3ds;
		}
        
		return null;
	}
	
	private double determinant(double[][] a) {
		int n = a.length;
		if (n == 2) 
			return a[0][0]*a[1][1] - a[1][0]*a[0][1];
		double m[][] = new double[n-1][n-1];
		double det = 0;
		for (int j1=0; j1<n; j1++) {
            for (int i=1; i<n; i++) {
            	int j2 = 0;
                for (int j=0; j<n; j++) {
                    if (j == j1) 
                    	continue;
                    m[i-1][j2] = a[i][j];
                    j2++;
                }
            }
            det = det + Math.pow(-1.0, j1)*a[0][j1]*determinant( m );
        }
		return det;
	}
	
	@Override
	public String toString() {
		return this.getClass().getSimpleName() + " [" + Arrays.toString(this.getTetrahedronPoints()) + "]";
	}
	
	public static void main(String args[]) {
		Point3D pi = new Point3D("1", 1,1,1);
		Point3D pj = new Point3D("2", 5,1,1);
		Point3D pk = new Point3D("3", 3,5,1);
		Point3D pl = new Point3D("4", 3,3,40);
		
		Tetrahedron t = new Tetrahedron(pi, pj, pk, pl);
		System.out.println(t.containsPoint(new Point3D("N", 3,4,1)));
	}
}