 /**********************************************************************
 * Copyright (C) by Michael Loesler, http://derletztekick.com           *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, see <http://www.gnu.org/licenses/>  *
 * or write to the                                                      *
 * Free Software Foundation, Inc.,                                      *
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
 *                                                                      *
  **********************************************************************/

package com.derletztekick.tools.geodesy.delaunay;

public class Sphere {
	private Point3D centerPoint;
	private double r=-1;
	private boolean isSphere = false;
	
	private Point3D[] spherePoints = new Point3D[4];
	
	public Sphere(Point3D pi, Point3D pj, Point3D pk, Point3D pl) throws ArithmeticException {
		this.init(pi, pj, pk, pl);
		if (!this.isSphere)
			throw new ArithmeticException(this.getClass().getSimpleName()+" Fehler, Punkte liegen *nicht* auf einer Kugel!\n" +
					pi + ",\n" +
					pj + ",\n" +
					pk);
	}
	
	public boolean isSphere() {
		//return (xk-xj)*(yj-yi) - (xj-xi)*(yk-yi) == 0;
		return this.isSphere;
	}
	
	public double getRadius() {
		return this.r;
	}
	
	public Point3D getCenter() {
		return this.centerPoint;
	}
	
	/**
	 * Ermittelt, ob ein Punkt p in der Kugel liegt
	 * In Kugel   == -1
	 * Auf Kugel  ==  0
	 * Ausserhalb ==  1
	 * 
	 * @param p
	 * @return inSphere
	 */
	public int containsPoint(Point3D p) {
		for (Point3D point : this.spherePoints)
			if (p.getId().equals(point.getId()))
				return 0;
		return (int)Math.signum(this.centerPoint.distance(p) - this.r);
	}
	
	// http://mysite.verizon.net/res148h4j/zenosamples/zs_sphere4pts.html
	private void init(Point3D pi, Point3D pj, Point3D pk, Point3D pl) {

	    double m11, m12, m13, m14;
	    double a[][] = new double[4][4];
	    
	    this.spherePoints = new Point3D[] {
				pi, pj, pk, pl
		};

	    for (int i=0; i<4; i++) {
	    	double x = this.spherePoints[i].getX();
	    	double y = this.spherePoints[i].getY();
	    	double z = this.spherePoints[i].getZ();
	    	
	        a[i][0] = x;
	        a[i][1] = y;
	        a[i][2] = z;
	        a[i][3] = 1.0;
	    }
	    m11 = this.determinant(a);

	    for (int i=0; i<4; i++) {
	    	double x = this.spherePoints[i].getX();
	    	double y = this.spherePoints[i].getY();
	    	double z = this.spherePoints[i].getZ();
	    	
	        a[i][0] = x*x + y*y + z*z;
	        a[i][1] = y;
	        a[i][2] = z;
	        a[i][3] = 1.0;
	    }
	    m12 = this.determinant(a);

	    for (int i=0; i<4; i++) {
	    	double x = this.spherePoints[i].getX();
	    	double y = this.spherePoints[i].getY();
	    	double z = this.spherePoints[i].getZ();
	    	
	        a[i][0] = x*x + y*y + z*z;
	        a[i][1] = x;
	        a[i][2] = z;
	        a[i][3] = 1.0;
	    }
	    m13 = this.determinant(a);

	    for (int i=0; i<4; i++) {
	    	double x = this.spherePoints[i].getX();
	    	double y = this.spherePoints[i].getY();
	    	double z = this.spherePoints[i].getZ();
	    	
	        a[i][0] = x*x + y*y + z*z;
	        a[i][1] = x;
	        a[i][2] = y;
	        a[i][3] = 1.0;
	    }
	    m14 = this.determinant(a);
	    
	    this.isSphere = m11 != 0; // Punkte in Ebene
	    
	    if (this.isSphere) {
	    	double xc =  0.5*m12/m11;
	        double yc = -0.5*m13/m11;
	        double zc =  0.5*m14/m11;
	    	this.centerPoint = new Point3D(this.getClass().getSimpleName(), xc, yc, zc);
	        
	        this.r  = this.centerPoint.distance(pi);
	    }
	    else
	    	this.spherePoints = new Point3D[4];
	}
	
	public Point3D[] getSpherePoints() {
		return this.spherePoints;
	}
	
	private double determinant(double[][] a) {
		int n = a.length;
		if (n == 2) 
			return a[0][0]*a[1][1] - a[1][0]*a[0][1];
		double m[][] = new double[n-1][n-1];
		double det = 0;
		for (int j1=0; j1<n; j1++) {
            for (int i=1; i<n; i++) {
            	int j2 = 0;
                for (int j=0; j<n; j++) {
                    if (j == j1) 
                    	continue;
                    m[i-1][j2] = a[i][j];
                    j2++;
                }
            }
            det = det + Math.pow(-1.0, j1)*a[0][j1]*determinant( m );
        }
		return det;
	}
	
	@Override
	public String toString() {
		return this.getClass().getSimpleName() + " [" + this.centerPoint.getX() + " / " + this.centerPoint.getY() + " / " + this.centerPoint.getZ() + "] r = " + this.r; 
	}

	public static void main(String args[]) {
		Point3D pi = new Point3D("1", 7,1,1);
		Point3D pj = new Point3D("2", 2,1,1);
		Point3D pk = new Point3D("3", 3,2,1);
		Point3D pl = new Point3D("4", 4,2,4);
		
		Sphere s = new Sphere(pi, pj, pk, pl);
		System.out.println(s);
	}
}
