 /**********************************************************************
 * Copyright (C) by Michael Loesler, http://derletztekick.com           *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, see <http://www.gnu.org/licenses/>  *
 * or write to the                                                      *
 * Free Software Foundation, Inc.,                                      *
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
 *                                                                      *
  **********************************************************************/

package com.derletztekick.tools.geodesy.delaunay;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

public class Delaunay {
	private List<Point2D> pointCloud = new ArrayList<Point2D>();
	private Set<Triangle2D> triangles = new LinkedHashSet<Triangle2D>();

	public Delaunay(List<Point2D> pointCloud) {
		this.pointCloud = pointCloud;
		init();
	}

	
	public Set<Triangle2D> getTriangles() {
		return this.triangles;
	}
	
	private void init() {
		if (this.pointCloud.size() < 3)
			return;

		Point2D extPoint1 = new Point2D("tmp1",  Integer.MAX_VALUE, 0.0); 
		Point2D extPoint2 = new Point2D("tmp2",  0.0,     Integer.MAX_VALUE); 
		Point2D extPoint3 = new Point2D("tmp3", -Integer.MAX_VALUE,-Integer.MAX_VALUE); 

		this.triangles.add(new Triangle2D(extPoint1, extPoint2, extPoint3));	
				
		while (this.pointCloud.size() > 0) {
			Point2D point = this.pointCloud.get(0);
			try {this.triangulate(point);} catch (ArithmeticException e) {e.printStackTrace();}
			this.pointCloud.remove(point);
		}
				
		Set<Triangle2D> triangles = new LinkedHashSet<Triangle2D>();		
		for (Triangle2D triangle : this.triangles) {
			if (triangle.containsPoint(extPoint1) < 1 || triangle.containsPoint(extPoint2) < 1 || triangle.containsPoint(extPoint3) < 1)
				continue;
			triangles.add(triangle);
		}
		this.triangles = triangles;		
	}
	
	private void triangulate(Point2D p) throws ArithmeticException {
		Triangle2D boundaryTriangle = null;
		int isInside = 1;
		for (Triangle2D triangle : this.triangles) {
			if ((isInside = triangle.containsPoint(p)) < 1) {
				boundaryTriangle = triangle;
				break;
			}
		}
		if (boundaryTriangle == null) {
			System.err.println(this.getClass().getSimpleName() + " boundaryTriangle == null");
			return;
		}
		// liegt auf einer Kante
		if (isInside == 0) {
			// Bestimme Kante, auf der der Punkt liegt
			Point2D[] edgePoints = boundaryTriangle.getEdge(p);
					
			if (edgePoints == null) 
				return;
			
			Point2D a = edgePoints[0];
			Point2D c = edgePoints[1];
			Point2D b = null;
			Point2D d = null;
			
			Triangle2D tri1 = null, tri2 = null, tri3 = null, tri4 = null;
			// Teile Dreieck auf
			Point2D[] triPoints = boundaryTriangle.getTrianglePoints();
			for (Point2D tmpD : triPoints) {
				if (tmpD.getId().equals(a.getId()) || tmpD.getId().equals(c.getId()))
					continue;
				d = tmpD;
				try {tri1 = new Triangle2D(p, d, a);} catch(ArithmeticException ae) { ae.printStackTrace(); }
				try {tri2 = new Triangle2D(p, d, c);} catch(ArithmeticException ae) { ae.printStackTrace(); }
								
				// Nachbarn anpassen bei neuen Dreiecken
				if (tri1 != null) {
					if (tri2 != null) 
						tri1.addNeighbour(tri2);

					Triangle2D tmp = boundaryTriangle.getNeighbour(d, a);
					if (tmp != null) {
						boundaryTriangle.removeNeighbour(tmp);
						tmp.removeNeighbour(boundaryTriangle);
						tri1.addNeighbour(tmp);
						tmp.addNeighbour(tri1);
					}
				}
				if (tri2 != null) {
					if (tri1 != null) 
						tri2.addNeighbour(tri1);

					Triangle2D tmp = boundaryTriangle.getNeighbour(d, c);
					if (tmp != null) {
						boundaryTriangle.removeNeighbour(tmp);
						tmp.removeNeighbour(boundaryTriangle);
						tri2.addNeighbour(tmp);
						tmp.addNeighbour(tri2);
					}
				}
				break;
			}

			Triangle2D neighbourTriangle = boundaryTriangle.getNeighbour(a, c);
			if (neighbourTriangle != null) {
				triPoints = neighbourTriangle.getTrianglePoints();
				for (Point2D tmpB : triPoints) {
					if (tmpB.getId().equals(a.getId()) || tmpB.getId().equals(c.getId()))
						continue;
					b = tmpB;
					try {tri3 = new Triangle2D(p, b, a);} catch(ArithmeticException ae) { ae.printStackTrace(); }
					try {tri4 = new Triangle2D(p, b, c);} catch(ArithmeticException ae) { ae.printStackTrace(); }
					
					// Nachbarn anpassen bei neuen Dreiecken
					if (tri3 != null) {
						if (tri1 != null) {
							tri3.addNeighbour(tri1);
							tri1.addNeighbour(tri3);
						}
						if (tri4 != null) 
							tri3.addNeighbour(tri4);
						
						Triangle2D tmp = neighbourTriangle.getNeighbour(b, a);
						if (tmp != null) {
							neighbourTriangle.removeNeighbour(tmp);
							tmp.removeNeighbour(neighbourTriangle);
							tri3.addNeighbour(tmp);
							tmp.addNeighbour(tri3);
						}
					}
					
					if (tri4 != null) {
						if (tri2 != null) {
							tri4.addNeighbour(tri2);
							tri2.addNeighbour(tri4);
						}
						
						if (tri3 != null)
							tri4.addNeighbour(tri3);
						
						Triangle2D tmp = neighbourTriangle.getNeighbour(b, c);
						if (tmp != null) {
							neighbourTriangle.removeNeighbour(tmp);
							tmp.removeNeighbour(neighbourTriangle);
							tri4.addNeighbour(tmp);
							tmp.addNeighbour(tri4);
						}
					}
					break;
				}
				neighbourTriangle.removeNeighbour(boundaryTriangle);
				boundaryTriangle.removeNeighbour(neighbourTriangle);
				this.triangles.remove(neighbourTriangle);
			}	
			this.triangles.remove(boundaryTriangle);
			neighbourTriangle = boundaryTriangle = null;
			
			// Fuege neue Dreiecke hinzu
			if (tri1 != null)
				this.triangles.add(tri1);
			if (tri2 != null)
				this.triangles.add(tri2);
			if (tri3 != null)
				this.triangles.add(tri3);
			if (tri4 != null)
				this.triangles.add(tri4);
			
			// Legalisiere neue Kanten
			if (tri1 != null)
				this.makeLegal(p, tri1, d, a, 0);
			if (tri2 != null)
				this.makeLegal(p, tri2, d, c, 0);
			if (tri3 != null)
				this.makeLegal(p, tri3, b, a, 0);
			if (tri4 != null)
				this.makeLegal(p, tri4, b, c, 0);
			
		}
		// Punkt liegt in einem Dreieck
		else {
			// Hole Punkte aus Dreieck
			Point2D[] triPoints = boundaryTriangle.getTrianglePoints();
			// erzeuge drei potenzielle (kleinere) Dreiecke
			Triangle2D tri1 = null, tri2 = null, tri3 = null;
			
			try {tri1 = new Triangle2D(p, triPoints[0], triPoints[1]); } catch(ArithmeticException ae) { ae.printStackTrace(); }
			try {tri2 = new Triangle2D(p, triPoints[0], triPoints[2]); } catch(ArithmeticException ae) { ae.printStackTrace(); }
			try {tri3 = new Triangle2D(p, triPoints[1], triPoints[2]); } catch(ArithmeticException ae) { ae.printStackTrace(); }
			
			// Fuege Nachbarn hinzu
			if (tri1 != null) {
				tri1.addNeighbour(tri2);
				tri1.addNeighbour(tri3);
			}
			
			if (tri2 != null) {
				tri2.addNeighbour(tri1);
				tri2.addNeighbour(tri3);
			}
			
			if (tri3 != null) {
				tri3.addNeighbour(tri1);
				tri3.addNeighbour(tri2);
			}
			
			// Teile vorhandene Nachbarn auf die neuen Dreiecke auf und
			// entferne das grosses Dreieck aus den Listen der Nachbarn
			Triangle2D neighbourTriangle = boundaryTriangle.getNeighbour(triPoints[0], triPoints[1]);
			if (neighbourTriangle != null) {
				neighbourTriangle.removeNeighbour(boundaryTriangle);
				neighbourTriangle.addNeighbour(tri1);
				if (tri1 != null)
					tri1.addNeighbour(neighbourTriangle);
			}
			neighbourTriangle = boundaryTriangle.getNeighbour(triPoints[0], triPoints[2]);
			if (neighbourTriangle != null) {
				neighbourTriangle.removeNeighbour(boundaryTriangle);
				neighbourTriangle.addNeighbour(tri2);
				if (tri2 != null)
					tri2.addNeighbour(neighbourTriangle);
			}
			neighbourTriangle = boundaryTriangle.getNeighbour(triPoints[1], triPoints[2]);
			if (neighbourTriangle != null) {
				neighbourTriangle.removeNeighbour(boundaryTriangle);
				neighbourTriangle.addNeighbour(tri3);
				if (tri3 != null)
					tri3.addNeighbour(neighbourTriangle);
			}
			
			// Entferne ursprngliches Dreieck aus der globalen Liste
			this.triangles.remove(boundaryTriangle);
			boundaryTriangle = null;
			// Fuege neue Dreiecke hinzu
			if (tri1 != null)
				this.triangles.add(tri1);
			if (tri2 != null)
				this.triangles.add(tri2);
			if (tri3 != null)
				this.triangles.add(tri3);
			
			// Legalisiere neue Kanten
			if (tri1 != null)
				this.makeLegal(p, tri1, triPoints[0], triPoints[1], 0);
			if (tri2 != null)
				this.makeLegal(p, tri2, triPoints[0], triPoints[2], 0);
			if (tri3 != null)
				this.makeLegal(p, tri3, triPoints[1], triPoints[2], 0);
		}
	}
	
	private void makeLegal(Point2D p, Triangle2D triangle, Point2D r, Point2D s, int counter) throws ArithmeticException {
		if (counter >= 1000)
			throw new ArithmeticException(this.getClass().getSimpleName() + ": Delaunay runs in an infinity loop! " + p +" "+ r +" "+ s);
		Triangle2D neighbourTriangle = triangle.getNeighbour(r, s);
		if (neighbourTriangle == null || !this.triangles.contains(triangle) || !this.triangles.contains(neighbourTriangle))
			return;
		Point2D[] neighbourTrianglePoints = neighbourTriangle.getTrianglePoints();
		
		Circle2D circumCircle = triangle.getCircumCircle();
		for (Point2D q : neighbourTrianglePoints) {
			if (q.getId().equals(r.getId()) || q.getId().equals(s.getId()))
				continue;
			// Punkt liegt im Kreis --> Kante unzulaessig
			if (circumCircle.containsPoint(q) < 0) {
				// Bilde zwei neue Dreiecke: bisher prs und qrs --> prq und psq
				Triangle2D tri1 = null, tri2 = null;
				
				try {tri1 = new Triangle2D(p, q, r);} catch(ArithmeticException ae) { ae.printStackTrace(); }
				try {tri2 = new Triangle2D(p, q, s);} catch(ArithmeticException ae) { ae.printStackTrace(); }
				// Korrigiere Nachbarn
				if (tri1 != null) {
					if (tri2 != null)
						tri1.addNeighbour(tri2);
					
					// QR-Nachbar fuer tri1
					Triangle2D tmp = neighbourTriangle.getNeighbour(q, r);
					if (tmp != null) {
						tmp.removeNeighbour(neighbourTriangle);
						neighbourTriangle.removeNeighbour(tmp);
						
						tri1.addNeighbour(tmp);
						tmp.addNeighbour(tri1);
					}
					
					// PR-Nachbar fuer tri1
					tmp = triangle.getNeighbour(p, r);
					if (tmp != null) {
						tmp.removeNeighbour(triangle);
						triangle.removeNeighbour(tmp);
						
						tri1.addNeighbour(tmp);
						tmp.addNeighbour(tri1);
					}
				}
				
				if (tri2 != null) {
					if (tri1 != null)
						tri2.addNeighbour(tri1);
					
					// QS-Nachbar fuer tri2
					Triangle2D tmp = neighbourTriangle.getNeighbour(q, s);
					if (tmp != null) {
						tmp.removeNeighbour(neighbourTriangle);
						neighbourTriangle.removeNeighbour(tmp);
						
						tri2.addNeighbour(tmp);
						tmp.addNeighbour(tri2);
					}
					// PS-Nachbar fuer tri2
					tmp = triangle.getNeighbour(p, s);
					if (tmp != null) {
						tmp.removeNeighbour(triangle);
						triangle.removeNeighbour(tmp);
						
						tri2.addNeighbour(tmp);
						tmp.addNeighbour(tri2);
					}
				}
				
				this.triangles.remove(triangle);
				this.triangles.remove(neighbourTriangle);
				
				if (tri1 != null)
					this.triangles.add(tri1);
				if (tri2 != null)
					this.triangles.add(tri2);
				
				if (tri1 != null)
					this.makeLegal(p, tri1, q, r, counter++);
				if (tri2 != null)
					this.makeLegal(p, tri2, q, s, counter++);
				
				neighbourTriangle.removeNeighbour(triangle);
				triangle.removeNeighbour(neighbourTriangle);
				
				triangle = neighbourTriangle = null; 
				break;
			}
		}	
	}
	
	/**
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		List<Point2D> pointCloud = new ArrayList<Point2D>();
//		pointCloud.add(new Point2D("1",  0, 0));
//		pointCloud.add(new Point2D("2", 10, 0));
//		pointCloud.add(new Point2D("3",  5, 2));
//		pointCloud.add(new Point2D("4",  6, 5));
//		pointCloud.add(new Point2D("5", 11, 4));
//		pointCloud.add(new Point2D("6",  1, 4));
//		pointCloud.add(new Point2D("7",  8,-7));
//		pointCloud.add(new Point2D("8",  1, 1));
//		pointCloud.add(new Point2D("9",  0, 1));
//		pointCloud.add(new Point2D("10",  1, 0));
//		pointCloud.add(new Point2D("11",  2, 2));
		
		
		pointCloud.add(new Point2D("1",  460848.2699857925,  8386439.694803107));
		pointCloud.add(new Point2D("2",  460848.2083907112,  8386439.5823420305));
		pointCloud.add(new Point2D("3",  460848.7814427460,  8386439.355585248));
		pointCloud.add(new Point2D("4",  460848.2084020019,  8386439.582338418));
		pointCloud.add(new Point2D("5",  460848.18750123336,  8386439.586799825));
		pointCloud.add(new Point2D("6",  460848.20976285805,  8386439.580300995));
		pointCloud.add(new Point2D("7",  460847.96765063494,  8386439.658463688));
		pointCloud.add(new Point2D("8",  460848.3786531337,  8386439.457219389));
		pointCloud.add(new Point2D("9",  460848.2083735584,  8386439.582349548));
		pointCloud.add(new Point2D("10", 460848.3177586797,  8386439.321601202));
		pointCloud.add(new Point2D("11", 460848.366648762,  8386439.390596199));
		pointCloud.add(new Point2D("12", 460979.61609641585,  8386611.524074238));
		pointCloud.add(new Point2D("13", 460884.1950411785,  8386546.22444283));
		pointCloud.add(new Point2D("14", 460880.37317979673,  8386542.136602185));
		pointCloud.add(new Point2D("15", 460847.90073647525,  8386444.423167387));
		pointCloud.add(new Point2D("16", 460844.81037602236,  8386439.899313713));
		
		double x0 = 0, y0 = 0;
		for (Point2D point : pointCloud) {
			double x = point.getX();
			double y = point.getY();
			x0 += x;
			y0 += y;
		}
		
		x0 /= pointCloud.size();
		y0 /= pointCloud.size();
		
		for (Point2D point : pointCloud) {
			point.setX(point.getX() - x0);
			point.setY(point.getY() - y0);
		}
		
		Delaunay delaunay = new Delaunay(pointCloud);
		Set<Triangle2D> triangles = delaunay.getTriangles();
		for (Triangle2D triangle : triangles) {
			System.out.println(triangle);
		}
	}
	
}
