 /**********************************************************************
 * Copyright (C) by Michael Loesler, http://derletztekick.com           *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, see <http://www.gnu.org/licenses/>  *
 * or write to the                                                      *
 * Free Software Foundation, Inc.,                                      *
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
 *                                                                      *
  **********************************************************************/

package com.derletztekick.tools.geodesy;

import no.uib.cipr.matrix.DenseMatrix;
import no.uib.cipr.matrix.DenseVector;
import no.uib.cipr.matrix.Matrices;
import no.uib.cipr.matrix.Matrix;
import no.uib.cipr.matrix.MatrixEntry;
import no.uib.cipr.matrix.UpperSymmPackMatrix;
import no.uib.cipr.matrix.Vector;

public class UpperSymmBlockMatrix extends UpperSymmSparseMatrix {
	private final int blockSize;
	public UpperSymmBlockMatrix(int size, int blockSize) {
		super(size);
		this.blockSize = blockSize;
	}
		
	public UpperSymmBlockMatrix(Matrix m, int blockSize) {
		super(m.numColumns());
		this.blockSize = blockSize;
		this.set(m);
	}
	
	public UpperSymmBlockMatrix(UpperSymmBlockMatrix m) {
		this(m, m.getBlockSize());
	}
	
	@Override
	public void set(int row, int column, double value) {
		if (this.blockSize > 0 && row/this.blockSize != column/this.blockSize)
			return;
		super.set(column,row,value);
	}
	
	@Override
	public int numSuperDiagonals() {
		return this.blockSize-1;
	}
	
	@Override
	public int numSubDiagonals() {
		return this.blockSize-1;
	}
	
	public int getBlockSize() {
		return this.blockSize;
	}
	
	@Override
	public Vector solve(Vector b, Vector x) {
		if (this.numRows != b.size())
            throw new IllegalArgumentException("numRows != b.size() (" + this.numRows + " != " + b.size() + ")");
        if (this.numColumns != x.size())
            throw new IllegalArgumentException("numColumns != x.size() (" + this.numColumns + " != " + x.size() + ")");

        for (int i=0; i<this.numColumns(); i+=this.blockSize) {
    		int indices[] = new int[this.blockSize];
    		for (int j=0; j<this.blockSize; j++)
    			indices[j] = i+j;

    		Vector subVec = new DenseVector(Matrices.getSubVector(b, indices));
    		Vector solVec = new DenseVector(this.blockSize);
    		Matrix subMat = new UpperSymmPackMatrix(Matrices.getSubMatrix(this, indices, indices));
    		subMat.solve(subVec, solVec);
    		
    		for (int j=0; j<this.blockSize; j++)
    			x.set(indices[j], solVec.get(j));
        }
		return x;
	}
	
	@Override
	public Matrix solve(Matrix B, Matrix X) {
		if (this.numRows != B.numRows())
            throw new IllegalArgumentException("numRows != B.numRows() (" + this.numRows + " != " + B.numRows() + ")");
        if (this.numColumns != X.numRows())
            throw new IllegalArgumentException("numColumns != X.numRows() (" + this.numColumns + " != " + X.numRows() + ")");
        if (X.numColumns() != B.numColumns())
            throw new IllegalArgumentException("X.numColumns() != B.numColumns() (" + X.numColumns() + " != " + B.numColumns() + ")");
		
		if (MathExtension.isIdentity(B)) {
			boolean isUpperSymmMatrix = X instanceof UpperSymmBlockMatrix || X instanceof UpperSymmPackMatrix;
			Matrix I = MathExtension.identity(this.blockSize);
	    	for (int i=0; i<this.numColumns(); i+=this.blockSize) {
	    		int indices[] = new int[this.blockSize];
	    		for (int j=0; j<this.blockSize; j++)
	    			indices[j] = i+j;

	    		Matrix subBlock    = new UpperSymmPackMatrix(Matrices.getSubMatrix(this, indices, indices));
	    		Matrix invSubBlock = new DenseMatrix(this.blockSize,this.blockSize);
	    		subBlock.solve(I, invSubBlock);
	    		
	    		for (int k=0; k<this.blockSize; k++)
	    			for (int l=isUpperSymmMatrix?k:0; l<this.blockSize; l++)
	    				X.set(indices[k], indices[l], invSubBlock.get(k,l));
	    	}
	    	return X;
		}
		else 
			return super.solve(B, X);
	}
	
	public static boolean isUpperSymmBlockMatrix(Matrix matrix, int blockSize) {
		if (matrix instanceof UpperSymmBlockMatrix)
			return ((UpperSymmBlockMatrix) matrix).getBlockSize() <= blockSize;
		
		for (MatrixEntry element : matrix)
			if (element.column() >= element.row() && (element.row()/blockSize != element.column()/blockSize) && element.get() != 0.0)
				return false;
		return true;
	}
	
	
//	public static void main(String args[]) {
//		UpperSymmBlockMatrix m = new UpperSymmBlockMatrix(6,2);
//		m.set(0,0, 10);
//		m.set(0,1, 20);
//		m.set(0,2, 30);
//		m.set(0,3, 40);
//		m.set(0,4, 50);
//		m.set(0,5, 60);
//		
//		m.set(1,0,-11);
//		m.set(1,1, 144);
//		m.set(1,2, 13);
//		m.set(1,3, 14);
//		m.set(1,4, 15);
//		m.set(1,5, 16);
//		
//		m.set(2,0, 21);
//		m.set(2,1, 22);
//		m.set(2,2, 32);
//		m.set(2,3, 24);
//		m.set(2,4, 25);
//		m.set(2,5, 26);
//		
//		m.set(3,0, 31);
//		m.set(3,1, 32);
//		m.set(3,2,-33);
//		m.set(3,3, 36);
//		m.set(3,4, 35);
//		m.set(3,5, 36);
//		
//		m.set(4,0, 41);
//		m.set(4,1, 42);
//		m.set(4,2, 43);
//		m.set(4,3, 44);
//		m.set(4,4, 36);
//		m.set(4,5,  2);
//		
//		m.set(5,5, 19);
//		
//		UpperSymmBlockMatrix n = new UpperSymmBlockMatrix(m);
//		UpperSymmBlockMatrix s = new UpperSymmBlockMatrix(m);
//		
//		MathExtension.print(m);
//		MathExtension.squareUpperSymmMatrix(MathExtension.cholInv(  MathExtension.chol(m, 1)));
//		MathExtension.print(m);
//		MathExtension.inv(n);
//		MathExtension.print(n);
//		
//		UpperSymmBlockMatrix inv = new UpperSymmBlockMatrix(m.numColumns, m.getBlockSize());
//		s.solve(MathExtension.identity(m.numColumns), inv);
//		MathExtension.print(inv);
//		
//		DenseVector vec = new DenseVector(new double[] {4, 6, 3, 1, 8, 6});
//		DenseVector sol = new DenseVector(vec.size());
//		s.solve(vec, sol);
//		MathExtension.print(sol);
//		
//		System.out.println(UpperSymmBlockMatrix.isUpperSymmBlockMatrix(s, 3));
//	}
}
