/**********************************************************************
 *                           MathExtension                              *
 ************************************************************************
 * Copyright (C) by Michael Loesler, http://derletztekick.com           *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, see <http://www.gnu.org/licenses/>  *
 * or write to the                                                      *
 * Free Software Foundation, Inc.,                                      *
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
 *                                                                      *
 **********************************************************************/

package com.derletztekick.tools.geodesy;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

import org.netlib.util.intW;

import com.github.fommil.netlib.LAPACK;

import no.uib.cipr.matrix.DenseMatrix;
import no.uib.cipr.matrix.DenseVector;
import no.uib.cipr.matrix.LowerSymmBandMatrix;
import no.uib.cipr.matrix.Matrices;
import no.uib.cipr.matrix.Matrix;
import no.uib.cipr.matrix.MatrixEntry;
import no.uib.cipr.matrix.MatrixSingularException;
import no.uib.cipr.matrix.NotConvergedException;
import no.uib.cipr.matrix.SVD;
import no.uib.cipr.matrix.UnitUpperTriangBandMatrix;
import no.uib.cipr.matrix.UnitLowerTriangBandMatrix;
import no.uib.cipr.matrix.UpperSymmBandMatrix;
import no.uib.cipr.matrix.UpperSymmPackMatrix;
import no.uib.cipr.matrix.Vector;
import no.uib.cipr.matrix.io.MatrixInfo;
import no.uib.cipr.matrix.io.MatrixSize;
import no.uib.cipr.matrix.io.MatrixVectorReader;
import no.uib.cipr.matrix.io.MatrixVectorWriter;
import no.uib.cipr.matrix.sparse.CompDiagMatrix;

/**
 * Klasse mit Erweiterungen der Math-Class von JAVA
 * @author Michael Loesler <derletztekick.com>
 *
 */
public final class MathExtension {
	/**
	 * Liefert in Abhaengigkeit vom Vorzeichen von b den Wert a positiv oder negativ.
	 * Das Vorzeichen von a wird ignoriert.
	 * @param a
	 * @param b
	 * @return sign
	 */
	public static final double SIGN (double a, double b) {
		return (b >= 0.0 ? Math.abs(a) : -Math.abs(a));
	}


	/**
	 * Winkelreduktion auf ein pos. Intervall
	 * Vergleich zur modularen Operation:
	 * <pre>-50%400 == -50</pre>
	 * <pre>mod(-50,400) == 350</pre>
	 *  
	 * @param x
	 * @param y
	 * @return mod
	 */
	public static final double MOD(double x, double y){
		return x-Math.floor(x/y)*y;
	}

	/**
	 * Liefert die Kondition einer Matrix c = cond(M) 
	 * mithilfe von SVD
	 * 
	 * [u v w] = svd(M)
	 * c = max(v)/min(v)
	 * 
	 * @param M
	 * @return c = cond(M)
	 * @throws NotConvergedException
	 */
	public static double cond(Matrix M) throws NotConvergedException {
		SVD uwv = SVD.factorize(M);
		double[] s = uwv.getS();
		// vgl. http://www.mathworks.de/help/techdoc/ref/cond.html
		int m = Math.min(M.numColumns(), M.numRows())-1;
		if (s[m]!=0)
			return s[0]/s[m];
		return 0;
	}	

	/**
	 * Liefert die Pseudoinverse Q = M<sup>+1</sup> 
	 * der Matrix M mithilfe von SVD
	 * 
	 * [u v w] = svd(M)
	 * Q = v*w<sup>-1</sup>*u<sup>T</sup> 
	 * 
	 * @param M
	 * @return Q = M<sup>+1</sup>
	 * @throws NotConvergedException
	 */
	public static Matrix pinv(Matrix M) throws NotConvergedException {
		return MathExtension.pinv(M, 0.0);
	}

	/**
	 * Liefert die Pseudoinverse Q = M<sup>+1</sup> 
	 * der Matrix M mithilfe von SVD
	 * 
	 * [u v w] = svd(M)
	 * Q = v*w<sup>-1</sup>*u<sup>T</sup> 
	 * 
	 * @param M
	 * @param tol
	 * @return Q = M<sup>+1</sup>
	 * @throws NotConvergedException
	 */
	public static Matrix pinv(Matrix M, double tol) throws NotConvergedException {
		SVD uwv = SVD.factorize(M);

		Matrix U  = uwv.getU();
		Matrix VT = uwv.getVt();
		double[] s = uwv.getS();
		//Matrix W = new DenseMatrix(VT.numColumns(), U.numColumns());
		Matrix W = new CompDiagMatrix(VT.numColumns(), U.numColumns());
		// Bestimme Toleranz neu
		// vgl. http://www.mathworks.de/help/techdoc/ref/pinv.html
		if (tol < Constant.EPS) {
			double norm2 = 0.0;
			for (int i=0; i<s.length; i++) 
				norm2 = Math.max(norm2, Math.abs(s[i]));

			tol = Math.max(M.numColumns(), M.numRows()) * norm2 * (tol < 0 ? Math.sqrt(Constant.EPS) : Constant.EPS);
		}

		for (int i=0; i<s.length; i++)
			if (Math.abs(s[i]) > tol)
				W.set(i,i, 1.0/s[i]);
		s = null;

		Matrix VW = new DenseMatrix(VT.numRows(), W.numColumns());
		VT.transAmult(W, VW);
		W = null;
		VT = null;

		Matrix Q = new DenseMatrix(M.numColumns(), M.numRows());
		VW.transBmult(U, Q);

		return Q;
	}

	/**
	 * Liefert die Pseudoinverse Q = M<sup>+1</sup> 
	 * der Matrix M mithilfe von SVD, wobei der Defekt
	 * der Matrix durch rank bereits vorgegeben ist.
	 * Es werden nur die ersten n Singulaerwerte beruecksichtigt.
	 * 
	 * [u v w] = svd(M)
	 * Q = v*w<sup>-1</sup>*u<sup>T</sup> 
	 * 
	 * @param M
	 * @param rank
	 * @return Q = M<sup>+1</sup>
	 * @throws NotConvergedException
	 */
	public static Matrix pinv(Matrix M, int rank) throws NotConvergedException {
		if (rank <= 0)
			return MathExtension.pinv(M, (double)rank);

		SVD uwv = SVD.factorize(M);

		Matrix U  = uwv.getU();
		Matrix VT = uwv.getVt();
		double[] s = uwv.getS();
		Matrix W = new CompDiagMatrix(VT.numColumns(), U.numColumns());
		//Matrix W = new DenseMatrix(VT.numColumns(), U.numColumns());
		// Korrigiere das Intervall auf 0 und Anz. Singul.werte
		rank = Math.max(0, Math.min(rank, s.length));

		for (int i=0; i<rank; i++)
			if (Math.abs(s[i]) > 0)
				W.set(i,i, 1.0/s[i]);
		s = null;

		Matrix VW = new DenseMatrix(VT.numRows(), W.numColumns());
		VT.transAmult(W, VW);
		W = null;
		VT = null;

		Matrix Q = new DenseMatrix(M.numColumns(), M.numRows());
		VW.transBmult(U, Q);

		return Q;
	}


	/**
	 * Liefert eine quadratische Einheitsmatrix der Dimension <code>size</code>
	 * @param size
	 * @return I
	 */
	public static Matrix identity(int size) {
		return new UnitUpperTriangBandMatrix(size,0);
	}

	/**
	 * Findet die Anzahl an besetzten Nebendiagonalen unterhalb der Hauptdiagonalen. 
	 * Hilfreich, wenn eine Matrix in einen anderen Typ konvertiert werden soll.
	 * @return numSubDiag
	 */
	public static int getNumSubDiagonals(Matrix A) {
		int kl = 0;
		for (MatrixEntry e : A)
			if (Math.abs(e.get()) != 0.0)
				kl = Math.max(kl, e.row() - e.column());
		return kl;
	}


	/**
	 * Findet die Anzahl an besetzten Nebendiagonalen ueberhalb der Hauptdiagonalen. 
	 * Hilfreich, wenn eine Matrix in einen anderen Typ konvertiert werden soll.
	 * @return numSuperDiag
	 */
	public static int getNumSuperDiagonals(Matrix A) {
		int ku = 0;
		for (MatrixEntry e : A)
			if (Math.abs(e.get()) != 0.0)
				ku = Math.max(ku, e.column() - e.row());
		return ku;
	}

	/**
	 * Liefert true, wenn die Matrix eine quadratische Einheitsmatrix ist.
	 * 
	 * @param m
	 * @return isIdentity
	 */
	public static boolean isIdentity(Matrix m) {
		if (!m.isSquare())
			return false;

		if (m instanceof UnitUpperTriangBandMatrix && ((UnitUpperTriangBandMatrix)m).numSuperDiagonals() == 0 || m instanceof UnitLowerTriangBandMatrix && ((UnitLowerTriangBandMatrix)m).numSubDiagonals() == 0) 
			return true;

		for (int i=0; i<m.numColumns(); i++) {
			if (m.get(i,i) != 1.0)
				return false;
		}

		if (m instanceof UpperSymmBandMatrix && ((UpperSymmBandMatrix)m).numSuperDiagonals() == 0 || m instanceof LowerSymmBandMatrix && ((LowerSymmBandMatrix)m).numSubDiagonals() == 0 || MathExtension.getNumSuperDiagonals(m) == 0 && MathExtension.getNumSubDiagonals(m) == 0)
			return true;

		for (MatrixEntry element : m) {
			int r = element.row();
			int c = element.column();
			double value = element.get();
			if (r == c && value != 1.0 || r != c && value != 0.0)
				return false;				
		}
		return true;
	}

	/**
	 * Loest das Gleichungssystem <code>N * x = n</code>. Der Vektor n wird hierbei mit dem Loesungsvektor <code>x</code> ueberschrieben. 
	 * Wenn <code>invert = true</code>, dann wird <code>N</code> mit dessen Inverse ueberschrieben.
	 * 
	 * @param N
	 * @param n
	 * @param invert
	 * @throws MatrixSingularException
	 * @throws IllegalArgumentException
	 */
	public static void solve(UpperSymmPackMatrix N, DenseVector n, boolean invert) throws MatrixSingularException, IllegalArgumentException {
		final String UPLO = "U";
		DenseMatrix nd = new DenseMatrix(n, false);
		double[] na = nd.getData();
		double Nd[] = N.getData();
		int numRows = N.numRows();
		int[] ipiv = new int[numRows];

		intW info = new intW(0);
		
		// http://www.netlib.org/lapack/double/dspsv.f
		LAPACK.getInstance().dspsv(UPLO, numRows, nd.numColumns(), Nd, ipiv, na, Math.max(1, numRows), info);

		if (info.val > 0)
			throw new MatrixSingularException();
		else if (info.val < 0)
			throw new IllegalArgumentException();

		if (invert) {
			double work[] = new double[numRows];

			// http://www.netlib.org/lapack/double/dsptri.f
			LAPACK.getInstance().dsptri(UPLO, numRows, Nd, ipiv, work, info);

			if (info.val > 0)
				throw new MatrixSingularException();
			else if (info.val < 0)
				throw new IllegalArgumentException();
		}
	}

	/**
	 * Liefert die Inverse einer symmetrischen oberen Dreiecksmatrix mittels <code>N = LDL'</code> Zerlegung. <code>N</code> wird hierbei ueberschrieben.
	 * 
	 * @param  N Matrix
	 * @throws MatrixSingularException
	 * @throws IllegalArgumentException
	 */
	public static void inv(UpperSymmPackMatrix N) throws MatrixSingularException, IllegalArgumentException {
		final String UPLO = "U";
		int numRows = N.numRows();
		int[] ipiv = new int[numRows];
		intW info = new intW(0);
		double qd[] = N.getData();

		// http://www.netlib.org/lapack/double/dsptrf.f
		LAPACK.getInstance().dsptrf(UPLO, numRows, qd, ipiv, info);

		if (info.val > 0)
			throw new MatrixSingularException();
		else if (info.val < 0)
			throw new IllegalArgumentException();

		double work[] = new double[numRows];

		// http://www.netlib.org/lapack/double/dsptri.f
		LAPACK.getInstance().dsptri(UPLO, numRows, qd, ipiv, work, info);

		if (info.val > 0)
			throw new MatrixSingularException();
		else if (info.val < 0)
			throw new IllegalArgumentException();
	}
	
	/**
	 * Bestimmt ausgewaehlte Eigenwerte einer symmetrischen oberen Dreiecksmatrix <code>N</code>. Die Indizes der zu bestimmeden
	 * Eigenwerte ergeben sich aus dem Intervall <code>il <= i <= iu</code>, mit <code>il >= 1</code> und <code>ul <= n</code>.
	 * Sie werden in aufsteigender Reihenfolge ermittelt. Ist die Flag <code>vectors = true</code> gesetzt, werden die zugehoerigen
	 * Eigenvektoren mitbestimmt. Durch die Flag <code>n</code> kann die Eigenwert/-vektorbestimmung auf die ersten <code>n</code>-Elemente
	 * begrenzt werden.
	 * 
	 * Die Eigenwerte <code>eval</code> werden als UpperSymmBandMatrix gespeichert, die Eigenvektoren <code>evec</code> in einer DenseMatrix.
	 * 
	 * HINWEIS: Die Matrix <code>N</code> wird bei dieser Zerlegung ueberschrieben!!!
	 * 
	 * @param N
	 * @param n
	 * @param il
	 * @param iu
	 * @param vectors
	 * @return {eval, evec}
	 * @throws NotConvergedException
	 * @throws IllegalArgumentException
	 */
	public static Matrix[] eig(UpperSymmPackMatrix N, int n, int il, int iu, boolean vectors) throws NotConvergedException, IllegalArgumentException {
		n = n < 0 ? N.numRows() : n;
		if (il < 1)
			throw new IllegalArgumentException("Fehler, unterer Eigenwertindex muss il >= 1: il = " + il);
		if (iu > n)
			throw new IllegalArgumentException("Fehler, oberer Eigenwertindex muss iu > n: iu = " + iu + ", n = " + n);
		
		final String jobz  = vectors ? "V" : "N";
		final String range = "I";
		final String uplo  = "U";
        
        double ap[] = N.getData();
        double vl = 0;
        double vu = 0;
        double abstol = 2.0 * LAPACK.getInstance().dlamch("S");
        intW m = new intW(0);
        UpperSymmBandMatrix eval = new UpperSymmBandMatrix(iu-il + 1, 0);
        DenseMatrix evec = vectors ? new DenseMatrix(iu-il + 1, n) : new DenseMatrix(0, 0);
        int ldz = Math.max(1,n);
        double work[] = new double[8*n];
        int iwork[] = new int[5*n];
        int ifail[] = vectors ? new int[n] : new int[0];
        intW info = new intW(0);
        
        if (il <= 0 || il > iu && n > 0 || iu > n)
        	throw new IllegalArgumentException();

		// http://www.netlib.org/lapack/double/dspevx.f
		LAPACK.getInstance().dspevx(jobz, range, uplo, n, ap, vl, vu, il, iu, abstol, m, eval.getData(), evec.getData(), ldz, work, iwork, ifail, info);
		
		if (info.val > 0)
            throw new NotConvergedException(NotConvergedException.Reason.Breakdown);
        else if (info.val < 0)
            throw new IllegalArgumentException("Fehler, Eingangsargumente fehlerhaft!");
		
		return new Matrix[] {
				eval, evec
		};
	}

	/**
	 * Liefert die Inverse einer symmetrischen (positiv definiten) Blockmatrix. Wendet das Verfahren nach Cholesky
	 * auf alle Blockmatrizen an zur Bestimmung von N = M<sup>-1</sup>. Eine Privotisierung findet nicht statt.
	 * 
	 * Von M wird nur die obere Dreiecksmatrix beruecksichtrigt. M wird durch M<sup>-1</sup> ueberschrieben!
	 *
	 * @param M
	 * @return inv(M)
	 * @see http://diegeodaeten.de/cholesky_zerlegung.html
	 * @see #chol
	 * @see #cholInv
	 * @see #squareUpperSymmMatrix
	 */
	@Deprecated
	public static Matrix inv(UpperSymmBlockMatrix M) {
		int bz = M.getBlockSize();
		for (int i=0; i<M.numColumns(); i+=bz) {
			int indices[] = new int[bz];
			for (int j=0; j<bz; j++)
				indices[j] = i+j;
			// Nutze "Sicht" in Matrix um Submatrix zu bekommen (erzeugt kein neues Objekt, sodass der Inhalt von M direkt ueberschrieben wird!) 
			Matrix subBlock = Matrices.getSubMatrix(M, indices, indices);
			MathExtension.squareUpperSymmMatrix(MathExtension.cholInv(MathExtension.chol(subBlock)));
		}
		return M;
	}

	/**
	 * Liefert die Zerlegungsmatrix nach der Cholesky-Faktorisierung M = R<sup>T</sup>R 
	 * M muss eine symmetrische positiv definite - wird nicht geprueft - Band-Matrix sein, wobei
	 * die p die Anzahl der Nebendiagonalen beziffert!
	 * Von M wird nur die obere Dreiecksmatrix beruecksichtrigt. M wird durch R ueberschrieben!
	 * 
	 * @param M
	 * qparam p 
	 * @return R
	 * @throws IllegalArgumentException
	 * @see http://diegeodaeten.de/cholesky_zerlegung.html
	 */
	@Deprecated
	public static Matrix chol(Matrix M, int p) {
		int n = M.numColumns();
		double tol = Math.sqrt(Constant.EPS);
		for (int j=0; j<n; j++) {
			int lam;
			for (int k = Math.max(0,j-p-1); k<j; k++) {
				lam = Math.min(k+p+1,n);
				for (int l=j; l<lam; l++) {
					M.set(j,l, M.get(j,l) - M.get(k,j)*M.get(k,l) );
				}
			}
			double ajj = M.get(j,j);
			ajj = ajj < tol ? tol : Math.sqrt(ajj);
			lam = Math.min(j+p+1,n);
			for (int l=j; l<lam; l++)
				M.set(j,l, M.get(j,l)/ajj );
		}
		return M;
	}

	/**
	 * Liefert die Zerlegungsmatrix nach der Cholesky-Faktorisierung M = R<sup>T</sup>R 
	 * M muss eine symmetrische positiv definite Matrix sein - wird nicht geprueft!
	 * Von M wird nur die obere Dreiecksmatrix beruecksichtrigt, sodass eine Obere-Symmetrische-Dreiecksmatrix
	 * erwartet wird. M wird durch R ueberschrieben!
	 * 
	 * @param M
	 * @return R
	 * @see http://diegeodaeten.de/cholesky_zerlegung.html
	 */
	@Deprecated
	public static Matrix chol(Matrix M)  {
		double tol = Math.sqrt(Constant.EPS);
		int n = M.numColumns();
		// Bestimme obere Dreiecksmatrix
		for (int j=0; j<n; j++) {
			double sum = M.get(j,j);
			for (int i=0; i<=j-1; i++) {
				sum -= M.get(i,j) * M.get(i,j);
			}
			if (Math.abs(sum)<tol)
				sum = tol;
			M.set(j,j, Math.sqrt(Math.abs(sum)));

			for (int k=j+1; k<n; k++) {
				sum = M.get(j,k);
				for (int i=0; i<=j-1; i++) {
					sum -= M.get(i,j) * M.get(i,k);
				}
				M.set(j,k, sum/M.get(j,j));
			}
		}
		return M;
	}

	/**
	 * Invertiert die aus der Choleskeyzerlegung stammende Matrix S=R<sup>-1</sup>
	 * 
	 * @param R
	 * @return S
	 * @see http://diegeodaeten.de/cholesky_zerlegung.html
	 */
	@Deprecated
	public static Matrix cholInv(Matrix R)  {
		int n = R.numColumns()-1;

		for (int j=n; j>=0; j--) {
			for (int k=n; k>=j+1; k--) {
				double sum = 0.0;
				for (int i=j+1; i<=k; i++)
					sum += R.get(j,i) * R.get(i,k);
				R.set(j,k, -sum/R.get(j,j));
			}
			R.set(j,j, 1.0/R.get(j,j));
		}
		return R;
	}

	/**
	 * Multipliziert eine obere Dreiecksmatrix mit sich selbst P = RR<sup>T</sup>
	 * 
	 * @param R
	 * @return P
	 * @see http://diegeodaeten.de/cholesky_zerlegung.html
	 */
	@Deprecated
	public static Matrix squareUpperSymmMatrix(Matrix R) {
		int n = R.numColumns();

		for (int j=0; j<n; j++) {
			for (int k=j; k<n; k++) {
				double sum = 0.0;
				for (int i=k; i<n; i++) 
					sum += R.get(j,i) * R.get(k,i);
				R.set(j,k, sum);
			}
		}
		return R;
	}

	/** 
	 * Druckt eine Matrix auf der Konsole aus
	 * @param M
	 */
	public static void print(Matrix M) {
		for (int i=0; i<M.numRows(); i++) {
			for (int j=0; j<M.numColumns(); j++) {
				System.out.print(M.get(i,j)+"  ");
			}
			System.out.println();
		}
	}

	/**
	 * Druckt einen Vektor auf der Konsole aus
	 * @param v
	 */
	public static void print(Vector v) {
		for (int i=0; i<v.size(); i++) {
			System.out.println(v.get(i));
		}
	}

	/**
	 * Invertiert eine Matrix mit privot-Suche
	 * @return A<sup>-1</sup>
	 * @throws MatrixSingularException
	 * @throws IllegalArgumentException
	 */
	@Deprecated
	public static Matrix inv(Matrix N) throws MatrixSingularException, IllegalArgumentException {
		if (N.numColumns() != N.numRows())
			throw new IllegalArgumentException(" Dimmensionsfehler: " + N.numRows() + " und " + N.numColumns());
		int n = N.numColumns();

		int permx[] = new int[n];
		int permy[] = new int[n];
		Matrix invM = new DenseMatrix(n,n);

		for (int i=0; i<n; i++) {
			permx[i] = -1;
			permy[i] = -1;
			for (int j=0; j<n; j++)
				invM.set(i, j, N.get(i,j));
		}

		for (int i=0; i<n; i++){
			double piv = 0.0;
			int nx=0, ny=0;
			for (int ix=0; ix<n; ix++)
				if (permx[ix] == -1) {
					for (int iy=0; iy<n; iy++)
						if (permy[iy] == -1 && Math.abs(piv) < Math.abs(invM.get(ix, iy))) {
							piv = invM.get(ix, iy);
							nx = ix;
							ny = iy;
						}
				}

			if (Math.abs(piv) < Constant.EPS){ // == 0.0
				throw new MatrixSingularException("Matrix is singular to working precision... " + Math.abs(piv));
			}
			else if (Math.abs(piv) < Math.sqrt(Constant.EPS)) {
				System.out.println("Warning: Matrix is close to singular or badly scaled... " + Math.abs(piv));
			}

			permx[nx] = ny;
			permy[ny] = nx;
			double tmp = 1.0 / piv;
			for (int j=0; j<n; j++)
				if (j!=nx) {
					double faktor = invM.get(j,ny) * tmp;
					for (int k=0; k<n; k++)
						invM.set(j, k, invM.get(j,k) - invM.get(nx,k) * faktor);
					invM.set(j, ny, faktor);
				}
			for (int j=0; j<n; j++)
				invM.set(nx, j, -tmp * invM.get(nx, j));
			invM.set(nx, ny, tmp);
		}

		for (int i=0; i<n; i++){
			for (int j=i; j<n; j++){
				if (permx[j]==i){
					for (int k=0; k<n; k++){
						double tmp = invM.get(i,k);
						invM.set(i, k, invM.get(j,k));
						invM.set(j, k, tmp);
					}
					permx[j] = permx[i];
					permx[i] = i;
					break;
				}
			}

			for (int j=i; j<n; j++){
				if (permy[j]==i){
					for (int k=0; k<n; k++){
						double tmp = invM.get(k,i);
						invM.set(k, i, invM.get(k,j));
						invM.set(k, j, tmp);
					}
					permy[j] = permy[i];
					permy[i] = i;
					break;
				}
			}
		}
		return invM;
	}

	/**
	 * Liefert das Kreuzprodukt zweier 3x1-Vektoren
	 * @param a
	 * @param b
	 * @return c
	 * @throws IllegalArgumentException Wenn die Anzahl der Elemente in a oder/und b ungleich 3
	 */
	public static DenseVector cross(DenseVector a, DenseVector b) throws IllegalArgumentException {
		if (a.size() != 3 || b.size() != 3)
			throw new IllegalArgumentException("Fehler, Kreuzprodukt nur fuer 3x1-Vektoren definiert. "+ a.size() +" und " + b.size());
		DenseVector c = new DenseVector(3);
		c.set(0, a.get(1)*b.get(2) - a.get(2)*b.get(1));
		c.set(1, a.get(2)*b.get(0) - a.get(0)*b.get(2));
		c.set(2, a.get(0)*b.get(1) - a.get(1)*b.get(0));
		return c;
	}

	/**
	 * Liefert die deserialisierte Matrix
	 * (Derzeit nur symmetrische, quadratische Matrizen)
	 * @param serialMatrix
	 * @return matrix
	 */
	public static Matrix getDeserializedMatrix(String serialMatrix) {
		Matrix M = null;
		if (serialMatrix != null && !serialMatrix.trim().isEmpty()) {
			MatrixVectorReader mvr = null;  
			try {
				mvr = new MatrixVectorReader(new StringReader( serialMatrix ));
				MatrixInfo infoIn = mvr.readMatrixInfo();
				MatrixSize sizeIn = mvr.readArraySize();
				if (sizeIn.isSquare() && infoIn.isSymmetric() && infoIn.isArray()) {
					int numEntrys = (sizeIn.numColumns()*sizeIn.numColumns()+sizeIn.numColumns())/2;
					double arr[] = new double[numEntrys+1];
					mvr.readArray(arr);
					M = new UpperSymmPackMatrix( sizeIn.numColumns() );
					for (int i=0, k=1; i<sizeIn.numRows(); i++) {
						for (int j=0; j<=i; j++) {
							M.set(j,i,arr[k++]);
						}
					}
				}
			} catch (IOException e) {
				e.printStackTrace();
			} finally {
				try {
					if (mvr != null)
						mvr.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return M;
	}

	/**
	 * Liefert die serialisierte Matrix
	 * (Derzeit nur symmetrische, quadratische Matrizen)
	 * @param matrix
	 * @return serialMatrix
	 */
	public static String getSerializedMatrix(Matrix matrix) {
		if (matrix != null && matrix instanceof UpperSymmPackMatrix) {
			StringWriter sw = new StringWriter();
			int numRows    = matrix.numRows();
			int numColumns = matrix.numColumns();
			int numEntrys  = (numRows*numColumns+numColumns)/2;

			MatrixInfo info = new MatrixInfo(false, MatrixInfo.MatrixField.Real, MatrixInfo.MatrixSymmetry.Symmetric);
			MatrixSize size = new MatrixSize(numRows, numColumns, numEntrys);
			MatrixVectorWriter out = new MatrixVectorWriter(sw);

			out.printMatrixInfo(info);
			out.printMatrixSize(size);

			out.printArray(((UpperSymmPackMatrix)matrix).getData());
			out.flush();
			out.close();

			// Replace ist notwendig, da MTJ beim Schreiben die Default-Locale-Einstellungen nutzt
			// und hierbei ggf. die Zahlen mit Komma statt Punkt rausschreibt. Beim Einlesen wird
			// die Locale-Einstellung nicht beachtet --> Exception
			// vgl. http://massapi.com/source/matrix-toolkits-java-read-only/src/no/uib/cipr/matrix/io/MatrixVectorWriter.java.html
			// vgl. http://code.google.com/p/matrix-toolkits-java/issues/detail?id=34
			return sw.toString().replace(',', '.');
		}
		return null;
	}
}