  /**********************************************************************
  *                          ConfidenceRegion                            *
  ************************************************************************
  * Copyright (C) by Michael Loesler, http://derletztekick.com           *
  *                                                                      *
  * This program is free software; you can redistribute it and/or modify *
  * it under the terms of the GNU General Public License as published by *
  * the Free Software Foundation; either version 3 of the License, or    *
  * (at your option) any later version.                                  *
  *                                                                      *
  * This program is distributed in the hope that it will be useful,      *
  * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
  * GNU General Public License for more details.                         *
  *                                                                      *
  * You should have received a copy of the GNU General Public License    *
  * along with this program; if not, see <http://www.gnu.org/licenses/>  *
  * or write to the                                                      *
  * Free Software Foundation, Inc.,                                      *
  * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
  *                                                                      *
   **********************************************************************/

package com.derletztekick.tools.geodesy;

import com.derletztekick.tools.geodesy.statistic.TestStatisticParameters;

import no.uib.cipr.matrix.DenseVector;
import no.uib.cipr.matrix.Matrices;
import no.uib.cipr.matrix.Matrix;
import no.uib.cipr.matrix.NotConvergedException;
import no.uib.cipr.matrix.SVD;
import no.uib.cipr.matrix.Vector;

public class ConfidenceRegion {
	private final Matrix coVar; 
	private final int dim;
	private double eigenvalues[], marginalValues[];
	private Matrix eigenvectors;
	private final int sortOrder[];
	private double helmertErrorEllipseAxes[] = new double[2];
	private double helmertErrorEllipseAngle = 0;
	private TestStatisticParameters testStatisticParameters;

	public ConfidenceRegion(TestStatisticParameters testStatisticParameters, Matrix coVar) throws IllegalArgumentException, NotConvergedException {
		this(coVar);
		this.testStatisticParameters = testStatisticParameters;
	}
	
	public ConfidenceRegion(Matrix coVar) throws IllegalArgumentException, NotConvergedException {
		if (!coVar.isSquare())
			throw new IllegalArgumentException(this.getClass() + " CoVar muss quadratisch sein!");
		
		for (int i=0; i<coVar.numColumns(); i++) {
			for (int j=0; j<coVar.numRows(); j++) {
				if (Double.isInfinite(coVar.get(i,j)) || Double.isNaN(coVar.get(i,j)) ) {
					throw new IllegalArgumentException(this.getClass()+" Matrix contains Inf or NaN Values! " +i+"x"+j+" "+coVar.get(i,j));
				}
			}
		}
		
		this.coVar   = coVar;
		this.dim     = coVar.numColumns();
		
		this.SVD();
		this.sortOrder = this.getSortOrder();
		
		this.calculateHelmertErrorEllipse();
	}
		
	/**
	 * Liefert die Halbachse der Ellipse/des Ellispoids
	 * @param index
	 * @return axis
	 */
	public double getConfidenceAxis(int index) {
		return Math.sqrt( this.getEigenvalue(index) * this.dim * this.getQuantile(this.dim) );
	}
	
	/**
	 * Bestimmt die Euler-Winkel aus den Eigenvektoren fuer die Sequenz R = Rx*Ry*Rz
	 * 
	 * Rx = [1    0        0
	 *       0 cos(rx) -sin(rx)
	 *       0 sin(rx)  cos(rx)]
	 * 
	 * Ry = [cos(ry) 0 sin(ry)
	 *         0     1    0
	 *      -sin(ry) 0 cos(ry)];
	 * 
	 * Rz = [cos(rz) -sin(rz) 0
	 *       sin(rz)  cos(rz) 0 
	 *         0          0   1];
	 * 
	 * 
	 * 
	 * @return [rx, ry, rz]
	 */
	public double[] getEulerAngles() {
		if (this.dim == 1)
			return new double[] {0, 0, 0};
		
		double rx, ry, rz, r11, r12, r13, r23, r33, r21, r22, r31, r32;
		rx = ry = rz = r11 = r12 = r13 = r23 = r33 = r21 = r22 = r31 = r32 = 0;

		r11 = this.eigenvectors.get(0, this.sortOrder[0]);
		r12 = this.eigenvectors.get(0, this.sortOrder[1]);
		
		r21 = this.eigenvectors.get(1, this.sortOrder[0]);
		r22 = this.eigenvectors.get(1, this.sortOrder[1]);
		
		if (this.dim == 2) {
			double det2 = r11*r22 - r12*r21;
			if (det2 < 0) {
				r11 = -r11;
				r21 = -r21;
			}
			rz = MathExtension.MOD( Math.atan2(-r12, r11), 2.0*Math.PI );
		}
		else if (this.dim == 3) {
			r13 = this.eigenvectors.get(0, this.sortOrder[2]);
			r23 = this.eigenvectors.get(1, this.sortOrder[2]);
			
			r31 = this.eigenvectors.get(2, this.sortOrder[0]);
			r32 = this.eigenvectors.get(2, this.sortOrder[1]);
			r33 = this.eigenvectors.get(2, this.sortOrder[2]);
						
			double det3 = r11*r22*r33 + r12*r23*r31 + r13*r21*r32 - r13*r22*r31 - r12*r21*r33 - r11*r23*r32;
			
			if (det3 < 0) {
				r11 = -r11;
				r21 = -r21;
				r31 = -r31;
			}
			
			rx = MathExtension.MOD( Math.atan2(-r23, r33), 2.0*Math.PI );
			ry = MathExtension.MOD( Math.atan2(r13, Math.hypot(r23, r33)), 2.0*Math.PI );
			rz = MathExtension.MOD( Math.atan2(-r12, r11), 2.0*Math.PI );
		}
		
		return new double[] {rx, ry, rz};
	}
	
	/**
	 * Liefert den Richtungskosinus des Eigenvektors <code>cos &phi;<sub>axis</sub> = dot(v,axis) / norm(v) / norm(axis)</code>,
	 * worin v der Eigenvektor und axis eine Koordinatensystemachse [1 0 0]<sup>T</sup>, [0 1 0]<sup>T</sup> oder [0 0 1]<sup>T</sup>
	 * @param indexOfEigenvalue
	 * @param indexOfCoordinateSystemAxis
	 * @return cos
	 */
	public double getDirectionCosine(int indexOfEigenvalue, int indexOfCoordinateSystemAxis) {
		Vector eigenvector = this.getEigenvector(indexOfEigenvalue);
		double norm = Math.sqrt( eigenvector.dot(eigenvector) );
		if (norm == 0)
			return 0.0;
		
		return Math.acos( eigenvector.get(indexOfCoordinateSystemAxis) / norm );
		
	}	
	
	/**
	 * Liefert den Eigenwert, Eigenwerte sind der Groesse nach geordnet
	 * @param index
	 * @return eigenvalue
	 */
	public double getEigenvalue(int index) {
		return this.eigenvalues[this.sortOrder[index]];
	}
	
	/**
	 * Liefert die Eigenvektoren, sind in Abhaengigkeit der Eigenwerte der Groesse nach geordnet
	 * @param index
	 * @return eigenvector
	 */
	public Vector getEigenvector(int index) {
		Vector eigenvector = new DenseVector(this.dim);
		for (int i=0; i<this.dim; i++) {
			eigenvector.set(i, this.eigenvectors.get(i, this.sortOrder[index]));
		}
		return eigenvector;
	}
	
	/**
	 * Liefert die Parameter der Grenzwertellipse/-elliposid.
	 * 
	 * R == Matrix der normierten Eigenvektoren
	 * v == Vektor der zugehoerigen Eigenwerte
	 * 
	 * m = Rv
	 * 
	 * @param index
	 * @return m
	 */
	public double getMarginalvalue(int index) {
		return this.marginalValues[index];
	}
	
	private void SVD() throws NotConvergedException {
		SVD uwv = SVD.factorize(this.coVar);
		this.eigenvectors = uwv.getU();
		this.eigenvalues  = uwv.getS();
 
		// Bestimme den maximalen Eigenwert und den zugehoerigen Index
		double maxEval = 0;
		int indexMaxEval = 0;
		
		for (int i=0; i<this.dim; i++) {
			double eval = Math.abs(this.eigenvalues[i]);
			if (eval > maxEval) {
				maxEval = eval;
				indexMaxEval = i;
			}
		}
		
		// erzeuge einen Punkt auf der Ellipse in Normallage in max. Richtung
		DenseVector maxEvalVecNormal = new DenseVector(this.dim);
		maxEvalVecNormal.set(indexMaxEval, Math.sqrt(maxEval));
		
		// rotiere Punkt auf tatsaechliche Lage der Ellipse
		DenseVector maxEvalVecRot = new DenseVector(this.dim);
		this.eigenvectors.mult(maxEvalVecNormal, maxEvalVecRot);		
		this.marginalValues = Matrices.getArray(maxEvalVecRot);
	}	
	
	/**
	 * Ermittelt die Sortierreihenfolge der Eigenwerte und speichert die Indizes in einem Array 
	 * @return order
	 */
	private final int[] getSortOrder() {
		int order[] = new int[this.dim];
		if (this.dim == 2) {
			if (this.eigenvalues[0] > this.eigenvalues[1]) {
				order[0] = 0;
				order[1] = 1;
			}
			else {
				order[0] = 1;
				order[1] = 0;
			}
		}
		if (this.dim == 3) {
			double maxValue = Math.max(this.eigenvalues[0], Math.max(this.eigenvalues[1], this.eigenvalues[2]));
			double minValue = Math.min(this.eigenvalues[0], Math.min(this.eigenvalues[1], this.eigenvalues[2]));
			for (int i=0; i<this.dim; i++) {
				if (this.eigenvalues[i] == maxValue) {
					order[0] = i;
				}
				else if (this.eigenvalues[i] == minValue) {
					order[2] = i;
				}
				else {
					order[1] = i;
				}
			}
		}
		return order;
	}
	
	/**
	 * Liefert die Halbachse der Fehlerellipse2D nach HELMERT
	 * @param index
	 * @return axis
	 */
	public double getConfidenceAxis2D(int index) {
		return this.getConfidenceAxis2D(index, true);
	}
	
	/**
	 * Liefert die Halbachse der Konfidenzellipse2D
	 * @param index
	 * @param isHelmertEllipse
	 * @return axis
	 */
	public double getConfidenceAxis2D(int index, boolean isHelmertEllipse) {
		return this.helmertErrorEllipseAxes[index] * (isHelmertEllipse?1.0:Math.sqrt((double)this.dim*this.getQuantile(this.dim)));
	}
	
	/**
	 * Liefert dem Drehwinkel der Fehlerellipse2D nach HELMERT
	 * @return angle
	 */
	public double getConfidenceAngle2D() {
		return this.helmertErrorEllipseAngle;
	}
	
	/**
	 * Berechnet fuer 2D und 3D Punkte die Fehlerellipse nach HELMERT
	 * fuer den Plot
	 */
	private void calculateHelmertErrorEllipse() {
		if (this.dim > 1) {
			double qxx = this.coVar.get(0, 0);
			double qyy = this.coVar.get(1, 1);
			double qxy = this.coVar.get(0, 1);
			double w = Math.sqrt( (qxx-qyy)*(qxx-qyy) + 4*qxy*qxy );
			double a = Math.sqrt(0.5*(qxx+qyy+w));
			double b = Math.sqrt(0.5*(qxx+qyy-w));
			this.helmertErrorEllipseAxes[0] = a;
			this.helmertErrorEllipseAxes[1] = b;
			//this.helmertErrorEllipseAngle = 0.5*MathExtension.MOD(Math.atan2(2.0*qxy, qxx-qyy), 2.0*Math.PI);
			this.helmertErrorEllipseAngle = MathExtension.MOD(0.5*Math.atan2(2.0*qxy, qxx-qyy), 2.0*Math.PI);
		}
		else { // 1D-Fall
			this.helmertErrorEllipseAxes[0] = Math.sqrt(this.coVar.get(0, 0));
			this.helmertErrorEllipseAxes[1] = 0.0;
			this.helmertErrorEllipseAngle = 0.0;
		}
	}
	
	/**
	 * Liefert das Quantil in abhaengigkeit der Dimension
	 * @param dim
	 * @return quantil
	 */
	private double getQuantile(int dim) {
		if (this.testStatisticParameters != null)
			return this.testStatisticParameters.getTestStatisticParameter(dim, Double.POSITIVE_INFINITY).getQuantile();		
		return 1.0;
	}
}
