 /**********************************************************************
 * Copyright (C) by Michael Loesler, http://derletztekick.com           *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, see <http://www.gnu.org/licenses/>  *
 * or write to the                                                      *
 * Free Software Foundation, Inc.,                                      *
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.            *
 *                                                                      *
  **********************************************************************/

package com.derletztekick.tools.geodesy;

public class Complex {
	public final static Complex ONE  = new Complex(1.0,0.0);
	public final static Complex ZERO = new Complex(0.0,0.0);
	
	private final double re, im;

	public Complex(double real) {
		this(real, 0);
	}
	
	public Complex(double real, double imag) {
		this.re = real;
		this.im = imag;
	}

	public double abs()   { 
		return Math.hypot(this.re, this.im); 
	}
	
	public double phase() { 
		return Math.atan2(this.im, this.re); 
	} 

	public Complex plus(Complex b) {
		return new Complex(this.re + b.re, this.im + b.im);
	}

	public Complex minus(Complex b) {
		return new Complex(this.re - b.re, this.im - b.im);
	}

	public Complex times(Complex b) {
		return new Complex(this.re * b.re - this.im * b.im, this.re * b.im + this.im * b.re);
	}
	
	public Complex divides(Complex b) {
		return this.times(b.reciprocal());
	}
	
	public Complex plus(double d) {
		return this.plus(new Complex(d));
	}

	public Complex minus(double d) {
		return this.minus(new Complex(d));
	}

	public Complex times(double d) {
		return this.times(new Complex(d));
	}
	
	public Complex divides(double d) {
		return this.divides(new Complex(d));
	}

	public Complex conjugate() {  
		return new Complex(this.re, -this.im); 
	}

	public Complex reciprocal() {
		double scale = this.re*this.re + this.im*this.im;
		return new Complex(re / scale, -this.im / scale);
	}

	public double real() {
		return re; 
	}
	
	public double image() { 
		return this.im; 
	}

	public Complex exp() {
		return new Complex(Math.exp(this.re) * Math.cos(this.im), Math.exp(this.re) * Math.sin(this.im));
	}

	public Complex sin() {
		return new Complex(Math.sin(this.re) * Math.cosh(this.im), Math.cos(this.re) * Math.sinh(this.im));
	}

	public Complex cos() {
		return new Complex(Math.cos(this.re) * Math.cosh(this.im), -Math.sin(this.re) * Math.sinh(this.im));
	}

	public Complex tan() {
		return this.sin().divides(this.cos());
	}
	
	public Complex sinh() {
		return new Complex(Math.sinh(this.re) * Math.cos(this.im), Math.cosh(this.re) * Math.sin(this.im));
	}
	
	public Complex cosh() {
		return new Complex(Math.cosh(this.re) * Math.cos(this.im), Math.sinh(this.re) * Math.sin(this.im));
	}
	
	public Complex tanh() {
		return this.sinh().divides(this.cosh());
	}

	public Complex coth() {
		return this.cosh().divides(this.sinh());
	}
	
	public Complex asin() {
		double re2 = this.re * this.re;
		double im2 = this.im * this.im;
		double t1 = re2 + im2;
		//double t2 = Math.sqrt( (t1-1.0) * (t1-1.0) + 4.0 * im2 );
		double t2 = Math.hypot( (t1-1.0), 2.0 * Math.abs(this.im) );
		return new Complex(
				0.5*Math.signum(this.re) * Math.acos(t2 - t1),
				//0.5*Math.signum(this.im) * MathExtension.acosh(t2 + t1),
				0.5*Math.signum(this.im) * Math.log((t2 + t1)+Math.sqrt((t2 + t1)*(t2 + t1) - 1.0))
		);
	}

	public Complex acos(){
		return (new Complex(0.5*Math.PI)).minus(this.asin());
	}
	
	public Complex log() {
		return new Complex(Math.log(this.abs()), this.phase());
	}
	
	public Complex pow(Complex z){
		return (z.times(this.log())).exp();
	}
	
	public Complex pow(double d){
		return this.pow(new Complex(d));
	}

		
	@Override
	public String toString() {
		return this.re + " " + (this.im < 0.0 ? "-" : "+") + " " + Math.abs(this.im) + "i";
	}
	
	public static void main(String args[]) {
		Complex a = new Complex(3.14, 5.78);

		System.out.println(a.pow(a));
	}
}
