/*
 * Decompiled with CFR 0.152.
 */
package com.derletztekick.geodesy.earth.ellipsoid;

import com.derletztekick.geodesy.earth.ellipsoid.EllipsoidType;
import com.derletztekick.geodesy.earth.point.PointXYZ;
import com.derletztekick.geodesy.earth.projection.point.PointLatLonH;

public class Ellipsoid {
    private final EllipsoidType ellType;
    private final double a;
    private final double b;

    public Ellipsoid() {
        this(EllipsoidType.GRS80);
    }

    public Ellipsoid(EllipsoidType type) {
        this.ellType = type;
        this.a = this.getMajorAxis();
        this.b = this.getMinorAxis();
    }

    public Ellipsoid(String name, double major, double minor) {
        this.ellType = EllipsoidType.USER_DEFINED;
        this.ellType.setLabel(name);
        this.a = major;
        this.b = minor;
    }

    public double getAdaptedGaussianSphere(double phi) {
        double a = this.getMajorAxis();
        double b = this.getMinorAxis();
        double c = a * a / b;
        double e2 = this.getE2Square();
        double V2 = 1.0 + e2 * Math.cos(phi) * Math.cos(phi);
        return Math.sqrt(c * c / V2 / V2);
    }

    public final double getMajorAxis() {
        return switch (this.ellType) {
            case EllipsoidType.BESSEL -> Math.exp(6.8046434637 * Math.log(10.0));
            case EllipsoidType.KRASSOWSKI -> 6378245.0;
            case EllipsoidType.HAYFORD -> 6378388.0;
            case EllipsoidType.GRS80, EllipsoidType.WGS84 -> 6378137.0;
            default -> this.a;
        };
    }

    public final double getMinorAxis() {
        return switch (this.ellType) {
            case EllipsoidType.BESSEL -> Math.exp(6.8031892839 * Math.log(10.0));
            case EllipsoidType.GRS80, EllipsoidType.WGS84 -> this.getMajorAxis() * Math.sqrt(1.0 - this.getE1Square());
            case EllipsoidType.KRASSOWSKI, EllipsoidType.HAYFORD -> this.getMajorAxis() * (1.0 - this.flattening());
            default -> this.b;
        };
    }

    public final double flattening() {
        double f = 0.0;
        switch (this.ellType) {
            case KRASSOWSKI: {
                f = 0.003352329869259135;
                break;
            }
            case HAYFORD: {
                f = 0.003367003367003367;
                break;
            }
            default: {
                f = (this.getMajorAxis() - this.getMinorAxis()) / this.getMajorAxis();
            }
        }
        return f;
    }

    public final double getE1Square() {
        double e1 = 0.0;
        switch (this.ellType) {
            case KRASSOWSKI: 
            case HAYFORD: {
                e1 = 2.0 * this.flattening() - this.flattening() * this.flattening();
                break;
            }
            case GRS80: {
                e1 = 0.0066943800229;
                break;
            }
            case WGS84: {
                e1 = 0.00669437999013;
                break;
            }
            default: {
                double a = this.getMajorAxis();
                double b = this.getMinorAxis();
                e1 = (a * a - b * b) / (a * a);
            }
        }
        return e1;
    }

    public final double getE2Square() {
        double e2 = 0.0;
        switch (this.ellType) {
            case KRASSOWSKI: 
            case HAYFORD: {
                e2 = this.flattening() * (2.0 - this.flattening()) / Math.pow(1.0 - this.flattening(), 2.0);
                break;
            }
            case GRS80: 
            case WGS84: {
                e2 = this.getE1Square() / (1.0 - this.getE1Square());
                break;
            }
            default: {
                double a = this.getMajorAxis();
                double b = this.getMinorAxis();
                e2 = (a * a - b * b) / (b * b);
            }
        }
        return e2;
    }

    private final double getN() {
        double n = 0.0;
        switch (this.ellType) {
            case KRASSOWSKI: 
            case HAYFORD: {
                n = this.flattening() / (2.0 - this.flattening());
                break;
            }
            default: {
                double a = this.getMajorAxis();
                double b = this.getMinorAxis();
                n = (a - b) / (a + b);
            }
        }
        return n;
    }

    public final double getMeridianArcLengthFromLatitude(double phi) {
        double a = this.getMajorAxis();
        double b = this.getMinorAxis();
        double n = this.getN();
        double alpha = 0.5 * (a + b) * (1.0 + 0.25 * Math.pow(n, 2.0) + 0.015625 * Math.pow(n, 4.0));
        double beta = -1.5 * n + 0.5625 * Math.pow(n, 3.0) - 0.09375 * Math.pow(n, 5.0);
        double gamma = 0.9375 * Math.pow(n, 2.0) - 0.46875 * Math.pow(n, 4.0);
        double delta = -0.7291666666666666 * Math.pow(n, 3.0) + 0.41015625 * Math.pow(n, 5.0);
        double epsilon = 0.615234375 * Math.pow(n, 4.0);
        return alpha * (phi + beta * Math.sin(2.0 * phi) + gamma * Math.sin(4.0 * phi) + delta * Math.sin(6.0 * phi) + epsilon * Math.sin(8.0 * phi));
    }

    public final double getLatitudeFromMeridianArcLength(double B) {
        double a = this.getMajorAxis();
        double b = this.getMinorAxis();
        double n = this.getN();
        double alpha = 0.5 * (a + b) * (1.0 + 0.25 * Math.pow(n, 2.0) + 0.015625 * Math.pow(n, 4.0));
        double beta = 1.5 * n - 0.84375 * Math.pow(n, 3.0) + 0.525390625 * Math.pow(n, 5.0);
        double gamma = 1.3125 * Math.pow(n, 2.0) - 1.71875 * Math.pow(n, 4.0);
        double delta = 1.5729166666666667 * Math.pow(n, 3.0) - 3.2578125 * Math.pow(n, 5.0);
        double epsilon = 2.142578125 * Math.pow(n, 4.0);
        double phi = B / alpha;
        return phi + beta * Math.sin(2.0 * phi) + gamma * Math.sin(4.0 * phi) + delta * Math.sin(6.0 * phi) + epsilon * Math.sin(8.0 * phi);
    }

    public PointLatLonH XYZ2LatLonH(PointXYZ pointXYZ) {
        double X = pointXYZ.getX();
        double Y = pointXYZ.getY();
        double Z = pointXYZ.getZ();
        double a = this.getMajorAxis();
        double b = this.getMinorAxis();
        double c = a * a / b;
        double e1Sqr = this.getE1Square();
        double e2Sqr = this.getE2Square();
        double p = Math.hypot(X, Y);
        double theta = Math.atan2(Z * a, p * b);
        double lon = Math.atan2(Y, X);
        double lat = Math.atan2(Z + e2Sqr * b * Math.pow(Math.sin(theta), 3.0), p - e1Sqr * a * Math.pow(Math.cos(theta), 3.0));
        double V = Math.sqrt(1.0 + e2Sqr * Math.cos(lat) * Math.cos(lat));
        double N = c / V;
        double hel = p / Math.cos(lat) - N;
        int i = 0;
        do {
            lat = Math.atan2(Z / p, 1.0 - N * e1Sqr / (N + hel));
            V = Math.sqrt(1.0 + e2Sqr * Math.cos(lat) * Math.cos(lat));
            N = c / V;
            hel = p / Math.cos(lat) - N;
        } while (i++ < 8);
        return new PointLatLonH(lat, lon, hel);
    }

    public PointXYZ LatLonH2XYZ(PointLatLonH point) {
        double lat = point.getLatitude();
        double lon = point.getLongitude();
        double hel = point.getHeight();
        double a = this.getMajorAxis();
        double b = this.getMinorAxis();
        double c = a * a / b;
        double e1Sqr = this.getE1Square();
        double e2Sqr = this.getE2Square();
        double V = Math.sqrt(1.0 + e2Sqr * Math.cos(lat) * Math.cos(lat));
        double N = c / V;
        double X = (N + hel) * Math.cos(lat) * Math.cos(lon);
        double Y = (N + hel) * Math.cos(lat) * Math.sin(lon);
        double Z = (N * (1.0 - e1Sqr) + hel) * Math.sin(lat);
        return new PointXYZ(X, Y, Z);
    }

    public String getName() {
        return this.ellType.getLabel();
    }

    public String toString() {
        return "Ellipsoid: (" + this.ellType.getLabel() + ") a = " + this.getMajorAxis() + ", b = " + this.getMinorAxis();
    }

    public EllipsoidType getType() {
        return this.ellType;
    }
}

